<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\ConsorzioTriveneto;

use Application\Admin\Form\FormConfig;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Localization;
use Application\Showcase\Model\Order;
use Application\Showcase\Payment\BasePayment;
use Application\Showcase\Payment\PaymentFailResponse;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\PaymentResponse;
use Application\Showcase\Payment\TicketResponse;
use Pongho\Core\TemplateResponse;
use Pongho\Form\Field\CheckboxField;
use Pongho\Form\Field\RadioField;
use Pongho\Form\Field\TextField;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Request;
use Pongho\Sdk\ConsorzioTriveneto\Cart;
use Pongho\Sdk\ConsorzioTriveneto\Client;
use Pongho\Sdk\ConsorzioTriveneto\Customer;
use Pongho\Sdk\ConsorzioTriveneto\Listener as ConsorzioTrivenetoListener;
use Pongho\Sdk\ConsorzioTriveneto\Merchant;
use Pongho\Sdk\ConsorzioTriveneto\PaymentInitRequest;

/**
 * Gestisce il pagamento tramite il gateway della banca Volksbank (Consorzio Triveneto S.p.a.).
 */
class Payment extends BasePayment
{
    /**
     * {@inheritdoc}
     */
    public function preparePayment()
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        $logger->debug(
            '[CONSORZIO TRIVENETO] Prepare order.',
            [
                'order'       => $this->order->getId(),
                'prev_status' => $this->order->status,
            ]
        );

        $this->order->convertCartToOrder(Order::STATUS_PAYMENT);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function handlePayment(PaymentOptions $options)
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        $options = $options->get($this->order);
        $url = $this->getPaymentUrl($options);

        $logger->debug(
            '[CONSORZIO TRIVENETO] Payment URL: ' . $url,
            [
                'order'   => $this->order->getId(),
                'options' => $options,
            ]
        );

        return new PaymentResponse(new RedirectResponse($url));
    }

    /**
     * {@inheritdoc}
     *
     * @todo Traduzione dei messaggi (potrebbero essere inseriti nel pannello di controllo del metodo di pagamento).
     */
    public function handleCancel($redirect_url)
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        $details = unserialize($this->order->payment_details);

        $logger->debug(
            '[CONSORZIO TRIVENETO][CANCEL] Handle cancel.',
            [
                'order'   => $this->order->getId(),
                'status'  => $this->order->status,
                'details' => $details,
            ]
        );

        if ($this->order->isPaid()) {
            $logger->notice(
                '[CONSORZIO TRIVENETO][CANCEL] Order paid.',
                [
                    'order'   => $this->order->getId(),
                    'status'  => $this->order->status,
                    'details' => $details,
                ]
            );

            throw new HttpNotFoundException();
        }

        if ($this->order->status === Order::STATUS_PAYMENT_AUTHORIZED) {
            $logger->notice(
                '[CONSORZIO TRIVENETO][CANCEL] Order payment authorized.',
                [
                    'order'   => $this->order->getId(),
                    'status'  => $this->order->status,
                    'details' => $details,
                ]
            );

            throw new HttpNotFoundException();
        }

        /**
         * @var \Pongho\Http\Request $request
         * @var \Pongho\Template\Theme $view
         */

        $request = $this->getContainer()->get('request');
        $view = $this->getContainer()->get('theme_view');

        if ($request->query->has('error')) {
            $logger->debug(
                '[CONSORZIO TRIVENETO][CANCEL] There was an error.',
                [
                    'order' => $this->order->getId(),
                    'error' => $request->query->get('error', 'NULL'),
                ]
            );

            if (isset($details['Error']) && $details['Error'] === $request->query->get('error')) {
                $this->order->status = Order::STATUS_CART;

                // Il messaggio sarà comunque riferito al caso in cui il pagamento è fallito.
                $message = 'Si è verificato un errore durante il pagamento.';

                if (isset($details['ErrorText'])) {
                    $message .= '<br><br><span style="color: red;">' . $details['ErrorText'] . '</span>';
                }

                $message .= '<br><br><a href="' . url($redirect_url) . '" class="btn">Torna al carrello</a>';

                $logger->notice(
                    '[CONSORZIO TRIVENETO][CANCEL] There was an error during checkout: ' . $details['ErrorText'],
                    [
                        'order'   => $this->order->getId(),
                        'status'  => $this->order->status,
                        'details' => $details,
                    ]
                );

                $view
                    ->setTemplate('message.php')
                    ->assignVars(
                        [
                            'message_type' => 'alert',
                            'title'        => 'Attenzione',
                            'message'      => $message,
                        ]
                    );
            } else {
                $this->handleCancelUnexpectedError();
            }
        } elseif (in_array($this->order->status, [Order::STATUS_CART, Order::STATUS_PAYMENT_FAILED])) {
            // Controllo lo stato 'payment_failed' per quando il pagamento fallisce
            // e lo stato 'cart' nel caso sia stata aggiornata la pagina.
            if ($this->order->status === Order::STATUS_CART) {
                // Mi assicuro di essere arrivato qui dopo il processo di pagamento.
                if (!is_array($details) || !isset($details['processed']) || !$details['processed']) {
                    $this->order->status = Order::STATUS_CART;

                    // Come diavolo sei arrivato qua? Torna al carrello!
                    $logger->warning(
                        '[CONSORZIO TRIVENETO][CANCEL] The order is a cart, and it’s not processed. WTF?',
                        [
                            'order'   => $this->order->getId(),
                            'status'  => $this->order->status,
                            'details' => $details,
                        ]
                    );

                    return new PaymentFailResponse(new RedirectResponse($redirect_url));
                }
            } else {
                // Riporto l’ordine allo stato di carrello, in modo che l’utente possa riprovare il pagamento.
                $this->order->status = Order::STATUS_CART;

                $logger->debug(
                    '[CONSORZIO TRIVENETO][CANCEL] Set status order to "cart".',
                    [
                        'order' => $this->order->getId(),
                    ]
                );
            }

            // Il messaggio sarà comunque riferito al caso in cui il pagamento è fallito.
            $message = 'Il pagamento non è andato a buon fine. Attendi una e-mail con l’esito della transazione prima'
                . ' di riprovare.<br><br><a href="' . url($redirect_url) . '" class="btn">Torna al carrello</a>';

            $logger->warning(
                '[CONSORZIO TRIVENETO][CANCEL] Payment failed.',
                [
                    'order'   => $this->order->getId(),
                    'status'  => $this->order->status,
                    'details' => $details,
                    '_POST'   => $_POST,
                    '_GET'    => $_GET,
                    '_SERVER' => $_SERVER,
                ]
            );

            $view
                ->setTemplate('message.php')
                ->assignVars([
                    'message_type' => 'alert',
                    'title'        => 'Attenzione',
                    'message'      => $message,
                ]);
        } else {
            $this->handleCancelUnexpectedError();
        }

        $response = new PaymentFailResponse(new TemplateResponse($view));

        $options = $this->order->payment->getHandlerOptions();

        $logger->debug(
            '[CONSORZIO TRIVENETO][CANCEL] Payment options: ',
            [
                'order'   => $this->order->getId(),
                'status'  => $this->order->status,
                'details' => $details,
                'options' => $options,
            ]
        );

        if ($options['notify_on_fail']) {
            $response->setSendNotifyToAdmin(true);
        }

        return $response;
    }

    protected function handleCancelUnexpectedError()
    {
        // C'è un qualche errore che ha impedito il corretto procedimento del pagamento.
        // Invio la notifica con i dettagli.

        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        $logger->critical(
            '[CONSORZIO TRIVENETO] Unexpected error.',
            [
                'order'   => $this->order->getId(),
                'status'  => $this->order->status,
                'details' => unserialize($this->order->payment_details),
                '_POST'   => $_POST,
                '_GET'    => $_GET,
                '_SERVER' => $_SERVER,
            ]
        );

        // Potrei riportare l'ordine allo stato di carrello, nel caso di errore,
        // ma in realtà non ho modo di sapere a cosa è dovuto il problema.
        // Vedi nella documentazione la sezione Capitolo 3 -> Messaggio NotificationMessage -> ErrorURL.

        // Mostro un messaggio all’utente.
        $message = 'Si è verificato un errore inatteso durante il pagamento, ma la transazione potrebbe essere andata'
            . ' comunque a buon fine. Ti consigliamo di controllare la tua e-mail per ulteriori dettagli sull’esito'
            . ' della transazione. I nostri tecnici sono già stati informati del problema.';

        $view = $this->getContainer()->get('theme_view');
        $view
            ->setTemplate('message.php')
            ->assignVars([
                'message_type' => 'alert',
                'title'        => 'Attenzione',
                'message'      => $message,
            ]);
    }

    /**
     * {@inheritdoc}
     */
    public function handleListener(Request $request, PaymentOptions $options)
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        if ($request->getMethod() !== 'POST') {
            $message = sprintf('HTTP Method %s not allowed.', $request->getMethod());
            $logger->warning('[CONSORZIO TRIVENETO] ' . $message);

            throw new \RuntimeException($message);
        }

        $order_id = (int) $request->query->get('o');

        /** @var Order $order */
        $order = Order::find($order_id);

        if ($order === null) {
            $message = 'Order not found.';
            $logger->warning('[CONSORZIO TRIVENETO] ' . $message);

            throw new \RuntimeException($message);
        }

        if ($order->payment->handler_class !== self::class) {
            $message = sprintf('Order not found with %s handler.', self::class);
            $logger->warning('[CONSORZIO TRIVENETO] ' . $message, ['order' => $order->getId()]);

            throw new \RuntimeException($message);
        }

        if ($order->isPaid()) {
            $message = 'Order paid.';
            $logger->warning('[CONSORZIO TRIVENETO] ' . $message, ['order' => $order->getId()]);

            throw new \RuntimeException($message);
        }

        $this->setOrder($order);

        $options = $options->get($order);

        $logger->debug('[CONSORZIO TRIVENETO] Order options', ['order' => $order->getId(), 'options' => $options]);

        if (!isset($options['cancel_url'])) {
            $message = 'Parameter "cancel_url" is required.';
            $logger->critical('[CONSORZIO TRIVENETO] ' . $message, ['order' => $order->getId()]);

            throw new \OutOfBoundsException($message);
        }

        if (!isset($options['ticket_url'])) {
            $message = 'Parameter "ticket_url" is required.';
            $logger->critical('[CONSORZIO TRIVENETO] ' . $message, ['order' => $order->getId()]);

            throw new \OutOfBoundsException($message);
        }

        $listener = new ConsorzioTrivenetoListener($logger);
        $listenerHandler = new ListenerHandler($order, $options['cancel_url'], $options['ticket_url'], $logger);

        return $listener->handle($request, $listenerHandler);
    }

    /**
     * {@inheritdoc}
     */
    public function handleTicket()
    {
        return new TicketResponse(true);
    }

    /**
     * {@inheritdoc}
     */
    public static function getFormFieldsConfig(FormConfig $config, Translator $translator, array $options = [])
    {
        return [
            'user'           => [
                'class' => TextField::class,
                'label' => $translator->trans('User'),
            ],
            'password'       => [
                'class' => TextField::class,
                'label' => $translator->trans('Password'),
            ],
            'transaction'    => [
                'class'   => RadioField::class,
                'label'   => $translator->trans('Transaction mode'),
                'options' => [
                    PaymentInitRequest::TRANSACTION_PURCHASE      => $translator->trans('Instant purchase'),
                    PaymentInitRequest::TRANSACTION_AUTHORIZATION => $translator->trans('Authorized purchase'),
                ],
            ],
            'notify_on_fail' => [
                'class'       => CheckboxField::class,
                'label'       => $translator->trans('Payment failed'),
                'description' => $translator->trans('Send a notification when the payment fails'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getDefaultOptions()
    {
        return [
            'user'           => '',
            'password'       => '',
            'notify_on_fail' => false,
            'transaction'    => PaymentInitRequest::TRANSACTION_PURCHASE,
        ];
    }

    /**
     * Restituisce l’indirizzo per il pagamento dell’ordine.
     *
     * Il metodo fa una richiesta (chiamata `PaymentInit`) tramite cURL al Payment Gateway di Consorzio Triveneto S.p.a.
     * In risposta riceve l’URL della Hosted Payment Page (HPP) e il PaymentID.
     *
     * @param array $options Elenco delle opzioni.
     * @throws \RuntimeException
     * @throws \OutOfBoundsException
     * @return string L’indirizzo del carrello per il pagamento.
     *
     * @todo Internazionalizzare il parametro `langid`.
     */
    protected function getPaymentUrl(array $options = [])
    {
        $order = $this->getOrder();

        if ($options['user'] == '') {
            throw new \OutOfBoundsException('Parameter "user" is required.');
        }

        if ($options['password'] == '') {
            throw new \OutOfBoundsException('Parameter "password" is required.');
        }

        $merchant = new Merchant($options['user'], $options['password'], $options['listener_url'], $options['cancel_url']);
        $cart = new Cart($order->total(), $this->getOrderTitle());
        $customer = new Customer($order->getCustomerEmail(), Customer::LANGUAGE_ITA);

        $client = new Client($this->isDebug());

        $request = new PaymentInitRequest($merchant, $cart, $customer);
        $request->useCustomerEmail();
        $request->setUdf1($order->payment_hash);
        $request->setTransactionAction($options['transaction']);

        /** @var \Pongho\Sdk\ConsorzioTriveneto\PaymentInitResponse $response */
        $response = $client->send($request);

        $details = ['paymentid' => $response->getPaymentID()];
        $order->payment_details = serialize($details);
        $order->save();

        return $response->getPaymentUrl();
    }

    /**
     * @return string
     *
     * @todo   Traduzione del titolo dell’ordine
     */
    protected function getOrderTitle()
    {
        $order = $this->getOrder();

        if ($this->isDebug()) {
            return sprintf('%d-%d', time(), $order->id);
        }

        return $order->id;
    }
}
