<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Payment\ConsorzioTriveneto;

use Application\Showcase\Model\Order;
use Application\Showcase\Payment\ListenerResponse;
use Pongho\Http\Request;
use Pongho\Http\Response;
use Pongho\Sdk\ConsorzioTriveneto\ListenerHandlerInterface;
use Pongho\Utilities\DateTime;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

class ListenerHandler implements ListenerHandlerInterface
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var string
     */
    protected $cancelUrl;

    /**
     * @var string
     */
    protected $ticketUrl;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @param string          $cancelUrl
     * @param string          $ticketUrl
     * @param LoggerInterface $logger
     */
    public function __construct(Order $order, $cancelUrl, $ticketUrl, ?LoggerInterface $logger = null)
    {
        $this->order = $order;
        $this->cancelUrl = $cancelUrl;
        $this->ticketUrl = $ticketUrl;
        $this->logger = $logger ?: new NullLogger();
    }

    /**
     * {@inheritdoc}
     */
    public function isVerifiedPaymentId(Request $request)
    {
        $requestPaymentId = $request->post->get('paymentid');

        $details = unserialize($this->order->payment_details);
        $orderPaymentId = is_array($details) && isset($details['paymentid']) ? $details['paymentid'] : '';

        $this->logger->debug('[CONSORZIO TRIVENETO] isVerifiedPaymentId', [
            'details'          => $details,
            'requestPaymentId' => $requestPaymentId,
            'orderPaymentId'   => $orderPaymentId,
        ]);

        return $requestPaymentId === $orderPaymentId;
    }

    /**
     * {@inheritdoc}
     */
    public function isProcessed(Request $request)
    {
        $details = unserialize($this->order->payment_details);
        $processed = is_array($details) && (isset($details['processed']) && (bool) $details['processed']);

        $this->logger->debug('[CONSORZIO TRIVENETO] isProcessed', [
            'details'   => $details,
            'processed' => $processed,
        ]);

        return $processed;
    }

    /**
     * {@inheritdoc}
     */
    public function isAuthorized(Request $request)
    {
        $requestHash = $request->post->get('udf1');
        $orderHash = $this->order->payment_hash;

        $this->logger->debug('[CONSORZIO TRIVENETO] isAuthorized', [
            'requestHash' => $requestHash,
            'orderHash'   => $orderHash,
        ]);

        return $requestHash === $orderHash;
    }

    /**
     * {@inheritdoc}
     */
    public function handleApproved(Request $request)
    {
        $this->logger->debug('[CONSORZIO TRIVENETO] handleApproved', [
            'request' => $request->__toString(),
        ]);

        $this->order->status = Order::STATUS_PAYMENT_AUTHORIZED;

        $details = $request->post->all();
        $details['processed'] = true;
        $this->order->payment_details = serialize($details);

        return new ListenerResponse(true, new Response('REDIRECT=' . $this->ticketUrl));
    }

    /**
     * {@inheritdoc}
     */
    public function handleCaptured(Request $request)
    {
        $this->logger->debug('[CONSORZIO TRIVENETO] handleCaptured', [
            'request' => $request->__toString(),
        ]);

        $this->order->status = Order::STATUS_ORDER;
        $this->order->paid_at = new DateTime();

        $details = $request->post->all();
        $details['processed'] = true;
        $this->order->payment_details = serialize($details);

        return new ListenerResponse(true, new Response('REDIRECT=' . $this->ticketUrl));
    }

    /**
     * {@inheritdoc}
     */
    public function handleError(Request $request, $result = null)
    {
        $this->order->status = Order::STATUS_PAYMENT_FAILED;
        $redirect_url = $this->cancelUrl;

        $details = $request->post->all();
        $this->order->payment_details = serialize($details);

        if (!$result && $request->post->has('Error')) {
            $query = [
                'error' => $request->post->get('Error'),
            ];

            $redirect_url .= (str_contains($redirect_url, '?') ? '&' : '?') . http_build_query($query, '', '&');
        }

        $this->logger->debug('[CONSORZIO TRIVENETO] handleError', [
            'request'  => $request->__toString(),
            'post'     => $request->post->all(),
            'result'   => $result,
            'redirect' => $redirect_url,
        ]);

        return new ListenerResponse(false, new Response('REDIRECT=' . $redirect_url));
    }
}
