<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use ActiveRecord\Base as Model;
use Application\Showcase\Utilities\Taxation;
use Pongho\Core\Kernel;
use Pongho\Utilities\DateTime;

/**
 * Modello per le taglie prodotto.
 *
 * @property int                                  $id
 * @property int                                  $node_id
 * @property int                                  $size_name_id
 * @property float                                $price
 * @property float                                $offer
 * @property float                                $weight
 * @property int                                  $availabilities
 * @property int                                  $bound_in_cart
 * @property \Pongho\Utilities\DateTime           $bound_in_cart_updated_at
 * @property int                                  $bound_in_payment
 * @property \Pongho\Utilities\DateTime           $bound_in_payment_updated_at
 * @property int                                  $bound_in_order
 * @property int                                  $shopping_points
 * @property \Application\Showcase\Model\SizeName $size_name
 * @property \Application\Showcase\Model\Node     $product
 * @property bool                                 $gs_enabled
 * @property string                               $gtin
 * @property string                               $mpn
 */
class Size extends Model
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'nodes_sizes';

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     */
    public static $belongs_to = [
        ['size_name', 'model' => SizeName::class],
        ['product', 'model' => Node::class, 'foreign_key' => 'node_id'],
    ];

    /**
     * {@inheritdoc}
     */
    public function save($perform_validation = true)
    {
        /** @var \Application\Showcase\Discount\DiscountCalculatorInterface $discountCalculator */
        $discountCalculator = Kernel::instance()->getContainer()->get('shop_discount_calculator');

        $this->offer = $discountCalculator->calculate(
            $this->price,
            $this->product->discount,
            $this->product->discount_type
        );

        return parent::save($perform_validation);
    }

    /**
     * Restituisce il nome della taglia.
     *
     * @return string
     */
    public function name()
    {
        if ($this->name) {
            return $this->name;
        } else {
            return $this->size_name->name();
        }
    }

    /**
     * Restituisce il prezzo reale della taglia senza sconti.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function realPrice(?Taxation $taxation = null)
    {
        return $taxation instanceof Taxation ? $taxation->calculate($this->price) : $this->price;
    }

    /**
     * Restituisce il prezzo reale formattato della taglia senza sconti.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatRealPrice(?Taxation $taxation = null)
    {
        return format_price($this->realPrice($taxation));
    }

    /**
     * Restituisce il prezzo della taglia.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function price(?Taxation $taxation = null)
    {
        return $taxation instanceof Taxation ? $taxation->calculate($this->offer) : $this->offer;
    }

    /**
     * Restituisce il prezzo formattato della taglia.
     *
     * @param \Application\Showcase\Utilities\Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatPrice(?Taxation $taxation = null)
    {
        return format_price($this->price($taxation));
    }

    /**
     * Restituisce la taglia in base all’ID del nodo e all’ID del nome della taglia.
     *
     * @param integer $node_id
     * @param integer $size_name_id
     * @return \Application\Showcase\Model\Size
     */
    public static function findByNodeIdAndSizeNameId($node_id, $size_name_id)
    {
        return self::first([
            'conditions' => [
                'node_id = :node AND size_name_id = :size_name',
                'node'      => $node_id,
                'size_name' => $size_name_id,
            ],
        ]);
    }

    /**
     * Restituisce le disponibilità considerando anche le righe negli ordini aperti.
     *
     * Indicando una riga d’ordine, questa verrà esclusa dalla somma delle taglie prodotto già occupate.
     *
     * @param \Application\Showcase\Model\OrderRow $row
     * @return integer
     */
    public function availabilities(?OrderRow $row = null)
    {
        return $this->availabilities - $this->boundAvailabilities($row);
    }

    /**
     * Restituisce la quantità di taglie prodotto occupate.
     *
     * Indicando una riga d’ordine, questa verrà esclusa dal conteggio.
     *
     * @param \Application\Showcase\Model\OrderRow $row
     * @return integer
     */
    public function boundAvailabilities(?OrderRow $row = null)
    {
        $cart_expiry = new DateTime(Node::BOUND_IN_CART_EXPIRY);
        $payment_expiry = new DateTime(Node::BOUND_IN_PAYMENT_EXPIRY);

        if ($row instanceof OrderRow) {
            if ($row->order->isOrder()) {
                $bound_conditions = [
                    "(o.status IN ('" . Order::STATUS_ORDER . "', '" . Order::STATUS_PENDING . "', '" . Order::STATUS_PAYMENT_AUTHORIZED . "'))",
                ];
            } else {
                $bound_conditions = [
                    "o.status = '" . Order::STATUS_CART . "' AND `from`.updated_at > :cart_expiry",
                    "o.status IN ('" . Order::STATUS_PAYMENT . "', '" . Order::STATUS_PAYMENT_LOCKED . "', '" . Order::STATUS_TRANSACTION . "') AND `from`.updated_at > :payment_expiry",
                    "o.status IN ('" . Order::STATUS_ORDER . "', '" . Order::STATUS_PENDING . "', '" . Order::STATUS_PAYMENT_AUTHORIZED . "')",
                ];

                $bound_conditions = [
                    '((' . implode(') OR (', $bound_conditions) . '))',
                    'cart_expiry'    => $cart_expiry,
                    'payment_expiry' => $payment_expiry,
                ];
            }

            $options = [
                'joins'      => 'INNER JOIN ' . Order::tableName() . ' AS o ON o.id = `from`.order_id',
                'conditions' => [
                    '`from`.size_id = :size AND `from`.id <> :row',
                    'size' => $this->id,
                    'row'  => $row->id,
                ],
            ];

            $options = OrderRow::addCondition($options, $bound_conditions);

            return OrderRow::sum('quantity', $options);
        }

        $bound = $this->bound_in_order;

        if ($this->bound_in_cart_updated_at > $cart_expiry) {
            $bound += $this->bound_in_cart;
        }

        if ($this->bound_in_payment_updated_at > $payment_expiry) {
            $bound += $this->bound_in_payment;
        }

        return $bound;
    }

    /**
     * Aggiorna le taglie occupate.
     *
     * Questo metodo viene lanciato in automatico in determinate occasioni (aggiunta o rimozione di un prodotto in
     * carrello, conferma di un ordine, evasione, etc). Deve tener conto di tre stadi possibili:
     *
     * - prodotto in carrello
     * - ordine in fase di pagamento
     * - ordine in attesa di evasione
     *
     * ATTENZIONE: questo metodo non salva nulla in database, ma aggiorna solo le proprietà del modello. Per tanto è
     * necessario eseguire un salvataggio manuale e la chainability ci viene in aiuto: `$size->updateBounds()->save()`.
     *
     * @return $this
     */
    public function updateBounds()
    {
        /**
         * Prodotti in carrello da un ora
         *
         * @var Size $data
         */
        $data = Size::first([
            'select'     => '`from`.id, SUM(r.quantity) AS bound_in_cart, MAX(r.updated_at) AS bound_in_cart_updated_at',
            'joins'      => 'INNER JOIN ' . OrderRow::tableName() . ' AS r ON r.size_id = `from`.id'
                         . ' INNER JOIN ' . Order::tableName() . ' AS o ON o.id = r.order_id',
            'conditions' => [
                'o.status = :status AND r.size_id = :size AND r.updated_at > :expiry',
                'status' => Order::STATUS_CART,
                'size'   => $this->id,
                'expiry' => new DateTime(Node::BOUND_IN_CART_EXPIRY),
            ],
            'group'      => '`from`.id',
        ]);

        $this->bound_in_cart = $data ? $data->bound_in_cart : 0;
        $this->bound_in_cart_updated_at = $data ? $data->bound_in_cart_updated_at : null;

        /**
         * Ordini in fase di pagamento da 2 ore
         *
         * @var Size $data
         */
        $data = Size::first([
            'select'     => '`from`.id, SUM(r.quantity) AS bound_in_payment, MAX(r.updated_at) AS bound_in_payment_updated_at',
            'joins'      => 'INNER JOIN ' . OrderRow::tableName() . ' AS r ON r.size_id = `from`.id'
                         . ' INNER JOIN ' . Order::tableName() . ' AS o ON o.id = r.order_id',
            'conditions' => [
                "o.status IN ('" . Order::STATUS_PAYMENT . "', '" . Order::STATUS_PAYMENT_LOCKED . "', '" . Order::STATUS_TRANSACTION . "') AND r.size_id = :size AND r.updated_at > :expiry",
                'size'   => $this->id,
                'expiry' => new DateTime(Node::BOUND_IN_PAYMENT_EXPIRY),
            ],
            'group'      => '`from`.id',
        ]);

        $this->bound_in_payment = $data ? $data->bound_in_payment : 0;
        $this->bound_in_payment_updated_at = $data ? $data->bound_in_payment_updated_at : null;

        /**
         * Ordini da evadere
         *
         * @var Size $data
         */
        $data = Size::first([
            'select'     => '`from`.id, SUM(r.quantity) AS bound_in_order',
            'joins'      => 'INNER JOIN ' . OrderRow::tableName() . ' AS r ON r.size_id = `from`.id'
                         . ' INNER JOIN ' . Order::tableName() . ' AS o ON o.id = r.order_id',
            'conditions' => [
                'o.status IN (:order, :pending, :authorized) AND r.size_id = :size',
                'order'      => Order::STATUS_ORDER,
                'pending'    => Order::STATUS_PENDING,
                'authorized' => Order::STATUS_PAYMENT_AUTHORIZED,
                'size'       => $this->id,
            ],
            'group'      => '`from`.id',
        ]);

        $this->bound_in_order = $data ? $data->bound_in_order : 0;

        return $this;
    }

    /**
     * @return $this
     */
    public function removeFromGoogleShoppingFeed()
    {
        $this->gs_enabled = false;

        return $this;
    }

    /**
     * @return int
     */
    public function getShoppingPoints()
    {
        return $this->shopping_points;
    }
}
