<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model;

use ActiveRecord\Base;

/**
 * @property int                             $id
 * @property string                          $handler_name
 * @property string                          $code
 * @property string                          $description
 * @property string                          $settings
 * @property string                          $email_sender_name
 * @property string                          $email_sender_email
 * @property string                          $email_subject
 * @property string                          $email_body
 * @property bool                            $email_sent
 * @property null|\Pongho\Utilities\DateTime $valid_from
 * @property \Pongho\Utilities\DateTime      $expire_on
 * @property OrderDiscountCondition          $condition
 * @property OrderDiscountCondition[]        $conditions
 */
class OrderDiscount extends Base
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'shop_order_discounts';

    /**
     * Relazione `has_many`.
     *
     * @var array
     */
    public static $has_many = [
        ['conditions', 'model' => OrderDiscountCondition::class, 'foreign_key' => 'discount_id'],
    ];

    /**
     * Relazione `has_one`.
     *
     * @var array
     */
    public static $has_one = [
        ['condition', 'model' => OrderDiscountCondition::class, 'foreign_key' => 'discount_id'],
    ];

    /**
     * @var array
     */
    protected $_settings;

    /**
     * Callback 'after_construct'.
     *
     * @var array
     */
    public static $after_construct = ['prepareSettingsOnCreate'];

    /**
     * Callback 'before_save'.
     *
     * @var array
     */
    public static $before_save = ['prepareSettingsOnSave'];

    /**
     * Cerca il buono sconto in base al codice.
     *
     * @param string $code Il codice del buono sconto.
     * @return OrderDiscount
     */
    public static function findByCode($code)
    {
        return static::first([
            'conditions' => ['code = ?', $code],
        ]);
    }

    /**
     * Cerca il primo sconto disponibile in base al nome dell’handler.
     *
     * @param string $handler_name Il nome dell’handler.
     * @return OrderDiscount
     */
    public static function findByHandler($handler_name)
    {
        return static::first([
            'conditions' => ['handler_name = ?', $handler_name],
        ]);
    }

    /**
     * Restituisce le condizioni associate allo sconto.
     *
     * @return \Application\Showcase\Model\OrderDiscountCondition[]
     */
    public function getConditions()
    {
        return OrderDiscountCondition::all([
            'conditions' => ['discount_id = ?', $this->id],
            'order'      => '`condition` DESC',
        ]);
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getHandlerName()
    {
        return $this->handler_name;
    }

    /**
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @param string $description
     * @return $this
     */
    public function setDescription($description)
    {
        $this->description = trim($description);

        return $this;
    }

    /**
     * @return string
     */
    public function getEmailSenderName()
    {
        return $this->email_sender_name;
    }

    /**
     * @param string $email_sender_name
     * @return $this
     */
    public function setEmailSenderName($email_sender_name)
    {
        $this->email_sender_name = trim($email_sender_name);

        return $this;
    }

    /**
     * @return string
     */
    public function getEmailSenderEmail()
    {
        return $this->email_sender_email;
    }

    /**
     * @param string $email_sender_email
     * @return $this
     */
    public function setEmailSenderEmail($email_sender_email)
    {
        $this->email_sender_email = trim($email_sender_email);

        return $this;
    }

    /**
     * @return string
     */
    public function getEmailSubject()
    {
        return $this->email_subject;
    }

    /**
     * @param string $email_subject
     * @return $this
     */
    public function setEmailSubject($email_subject)
    {
        $this->email_subject = trim($email_subject);

        return $this;
    }

    /**
     * @return string
     */
    public function getEmailBody()
    {
        return $this->email_body;
    }

    /**
     * @param string $email_body
     * @return $this
     */
    public function setEmailBody($email_body)
    {
        $this->email_body = trim($email_body);

        return $this;
    }

    /**
     * @return bool
     */
    public function isReadyToSendNotification()
    {
        return $this->getEmailSenderName() && $this->getEmailSenderEmail() && $this->getEmailSubject() && $this->getEmailBody();
    }

    /**
     * @return bool
     */
    public function isEmailSent()
    {
        return $this->email_sent;
    }

    /**
     * @return $this
     */
    public function markAsNotified()
    {
        $this->email_sent = true;

        return $this;
    }

    /**
     * @return null|\Pongho\Utilities\DateTime
     */
    public function getValidFrom()
    {
        return $this->valid_from;
    }

    /**
     * @return \Pongho\Utilities\DateTime
     */
    public function getExpireOn()
    {
        return $this->expire_on;
    }

    /**
     * Indica se è possibile eliminare lo sconto.
     *
     * Uno sconto non può essere eliminato quando è già stato associato ad almeno un ordine.
     *
     * @return bool
     */
    public function isDeletable()
    {
        $options = [
            'conditions' => ['discount_id = ?', $this->id],
        ];

        return OrderOrderDiscount::count($options) === 0;
    }

    /**
     * @internal
     */
    public function prepareSettingsOnCreate()
    {
        $settings = json_decode($this->settings, true);
        $this->_settings = is_array($settings) ? $settings : [];
    }

    /**
     * @internal
     */
    public function prepareSettingsOnSave()
    {
        $this->settings = json_encode($this->_settings);
    }

    /**
     * @param string $name
     * @param mixed  $default
     * @return mixed
     */
    public function getSetting($name, $default = null)
    {
        return array_key_exists($name, $this->_settings) ? $this->_settings[$name] : $default;
    }

    /**
     * @param string $name
     * @param mixed  $value
     * @return $this
     */
    public function setSetting($name, $value)
    {
        $this->_settings[$name] = $value;

        return $this;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function delSetting($name)
    {
        unset($this->_settings[$name]);

        return $this;
    }
}
