<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Model\Manager;

use Application\Core\Entity\LanguageSiteInterface;
use Application\Core\Model\File;
use Application\Showcase\GoogleShopping\Product;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Driver\Exception as DBALDriverException;
use Doctrine\DBAL\Exception as DBALException;
use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Statement;
use PDO;

class GoogleShoppingProductManager
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @var LanguageSiteInterface
     */
    protected $language_site;

    public function __construct(Connection $connection, LanguageSiteInterface $language_site)
    {
        $this->connection = $connection;
        $this->language_site = $language_site;
    }

    /**
     * @return Product[]
     * @throws DBALException
     * @throws DBALDriverException
     */
    public function findAll(?array $order = null, ?int $limit = null, ?int $offset = null): array
    {
        $sth = $this->prepareStatement($order, $limit, $offset);
        $result = $sth->executeQuery();

        $products = [];
        foreach ($result->fetchAllAssociative() as $row) {
            $product = Product::createFromArray($row);
            $product->prepare($this->language_site->getSite());

            $products[] = $product;
        }

        $this->populateTerms($products);
        $this->populateImages($products);

        return $products;
    }

    /**
     * @throws DBALDriverException
     * @throws DBALException
     */
    public function countAll(): int
    {
        return (int) $this->prepareStatement(null, null, null, true)
            ->executeQuery()
            ->fetchOne();
    }

    /**
     * @throws DBALException
     */
    protected function prepareStatement(?array $order = null, ?int $limit = null, ?int $offset = null, bool $count = false): Statement
    {
        $query = $this->buildQuery();

        if ($count) {
            $query = "SELECT COUNT(*) AS n FROM ($query) AS t";
        } else {
            if ($order) {
                $order = $this->validateOrder($order);

                if ($order) {
                    $query .= ' ORDER BY ' . implode(', ', $order);
                }
            }

            if ($limit) {
                $query = $this->connection->getDatabasePlatform()->modifyLimitQuery($query, $limit, $offset);
            }
        }

        $sth = $this->connection->prepare($query);
        $sth->bindValue('enabled', true, ParameterType::BOOLEAN);
        $sth->bindValue('site', $this->language_site->getSiteId(), ParameterType::INTEGER);
        $sth->bindValue('language', $this->language_site->getLanguageId(), ParameterType::INTEGER);

        return $sth;
    }

    protected function buildQuery(): string
    {
        return <<<SQL
SELECT
  sizes.id,
  products_translations.title,
  products_translations.content AS description,
  products_translations.permalink AS link,
  sizes.price,
  products.discount,
  products.discount_type,
  sizes.availabilities,
  sizes.gtin,
  sizes.mpn,
  products.id AS item_group_id,
  products.module_site_id,
  products.gs_color AS color,
  products.gs_gender AS gender,
  products.gs_age_group AS age_group,
  products.gs_material AS material,
  products.gs_pattern AS pattern,
  sizes_names.name AS size,
  products.featured_image_id
FROM pongho_nodes_sizes AS sizes
  INNER JOIN pongho_nodes AS products ON products.id = sizes.node_id
  INNER JOIN pongho_modules_sites AS ms ON ms.id = products.module_site_id
  INNER JOIN pongho_nodes_translations AS products_translations ON products_translations.node_id = products.id
    AND products_translations.language_id = :language
  INNER JOIN pongho_nodes_sizes_names AS sizes_names ON sizes_names.id = sizes.size_name_id
    AND COALESCE(sizes_names.size_type_id, 0) = COALESCE(products.size_type_id, 0)
WHERE products.status = 'publish'
  AND products.published_at < NOW()
  AND ms.site_id = :site
  AND sizes.gs_enabled = :enabled
SQL;
    }

    protected function validateOrder(array $order): array
    {
        $map = [
            'title'          => 'products_translations.title',
            'product_type'   => 'categories_translations.title',
            'brand'          => 'brands_translations.title',
            'availabilities' => 'sizes.availabilities',
        ];

        $sort = [];

        foreach ($order as $key => $asc_or_desc) {
            if (isset($map[$key])) {
                $sort[] = $map[$key] . ' ' . ($asc_or_desc === 'DESC' ? 'DESC' : 'ASC');
            }
        }

        return $sort;
    }

    /**
     * @param Product[] $products
     * @throws DBALException
     * @throws DBALDriverException
     */
    protected function populateTerms(array $products)
    {
        if ($products === []) {
            return;
        }

        $items = [];

        foreach ($products as $item) {
            $product_id = $item->getProductId();
            $items[$product_id][] = $item;
        }

        $product_ids = implode(',', array_map('intval', array_keys($items)));

        $query = <<<SQL
SELECT
  nt.node_id AS product_id,
  terms.taxonomy,
  terms.gs_category,
  terms_translations.title
FROM pongho_nodes_terms AS nt
  LEFT JOIN pongho_terms AS terms ON terms.id = nt.term_id
  LEFT JOIN pongho_terms_translations AS terms_translations ON terms_translations.term_id = terms.id
WHERE nt.node_id IN ($product_ids)
  ANd terms.taxonomy IN ('category', 'brand')
  AND terms_translations.language_id = :language
SQL;

        $sth = $this->connection->prepare($query);
        $sth->bindValue('language', $this->language_site->getLanguageId(), ParameterType::INTEGER);

        foreach ($sth->executeQuery()->fetchAllAssociative() as $row) {
            foreach ($items[$row['product_id']] as $item) {
                if ($row['taxonomy'] === 'category') {
                    $item->setProductType($row['title']);
                    $item->setGoogleCategory($row['gs_category']);
                }

                if ($row['taxonomy'] === 'brand') {
                    $item->setBrand($row['title']);
                }
            }
        }
    }

    /**
     * @param Product[] $products
     * @throws DBALException
     */
    protected function populateImages(array $products)
    {
        if ($products === []) {
            return;
        }

        $items = [];
        $featured_images = [];
        $additional_images = [];

        foreach ($products as $item) {
            $product_id = $item->getProductId();
            $items[$product_id][] = $item;

            $featured_images[$item->getFeaturedImageId()][] = $product_id;
        }

        $product_ids = implode(',', array_map('intval', array_keys($items)));

        if ($this->connection->getDatabasePlatform()->getName() === 'postgresql') {
            $condition = "CAST(gallery.identifier AS integer) IN ($product_ids)";
        } else {
            $condition = "CAST(gallery.identifier AS SIGNED) IN ($product_ids)";
        }

        $query = <<<SQL
SELECT
  gallery.identifier AS product_id,
  images.file_id
FROM pongho_gallery_images AS images
  INNER JOIN pongho_gallery AS gallery ON gallery.id = images.gallery_id
WHERE $condition
SQL;

        foreach ($this->connection->fetchAllAssociative($query) as $row) {
            $additional_images[$row['file_id']][] = $row['product_id'];
        }

        $image_ids = array_merge(array_keys($additional_images), array_keys($featured_images));
        $options = ['conditions' => ['id IN (' . implode(',', array_map('intval', $image_ids)) . ')']];

        /** @var File $file */
        foreach (File::all($options) as $file) {
            $image = src($file, 'googleshopping');

            if (isset($additional_images[$file->id])) {
                foreach ($additional_images[$file->id] as $product_id) {
                    foreach ($items[$product_id] as $item) {
                        $item->addAdditionalImage($image);
                    }
                }
            }

            if (isset($featured_images[$file->id])) {
                foreach ($featured_images[$file->id] as $product_id) {
                    foreach ($items[$product_id] as $item) {
                        $item->setImage($image);
                    }
                }
            }
        }
    }

    /**
     * @param Product[] $products
     * @return Product[]
     * @throws DBALException
     */
    public function populateNodeType(array $products): array
    {
        if ($products === []) {
            return $products;
        }

        $module_site_ids = [];

        foreach ($products as $product) {
            $module_site_ids[] = $product->getModuleSiteId();
        }

        $module_site_ids = implode(',', array_unique($module_site_ids));

        $query = <<<SQL
SELECT ms.id AS module_site_id, m.node_type
FROM pongho_modules_sites AS ms
  INNER JOIN pongho_modules AS m On m.id = ms.module_id
WHERE ms.id IN ($module_site_ids)
SQL;

        $node_types = [];
        foreach ($this->connection->fetchAllAssociative($query) as $row) {
            $node_types[$row['module_site_id']] = $row['node_type'];
        }

        foreach ($products as $product) {
            $product->setNodeType($node_types[$product->getModuleSiteId()]);
        }

        return $products;
    }
}
