<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Installer;

use Application\Core\Installer\Installer;
use Application\Core\Model\Permit;
use Application\Core\Model\Role;
use Application\Showcase\Kernel;

class ShopInstaller extends Installer
{
    public static $controllers = [
        \Application\Showcase\Controller\ShopController::class => [
            'path'     => '/shop/',
            'name'     => 'Shop',
            'position' => 90,
        ],
    ];

    /**
     * Crea la struttura e i record di supporto
     */
    public function install()
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->container->get('logger');

        $logger->info('[Showcase][Shop][Installer] Creazione delle tabelle dello Shop');

        $schema = $this->getSchema();

        // Metodi di spedizione
        $shippings_table = $schema->createTable('pongho_shippings');
        $shippings_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $shippings_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
        $shippings_table->addColumn('description', 'text', ['length' => 65535, 'default' => '']);

        $shippings_table->setPrimaryKey(['id']);

        // Condizioni per i metodi di spedizione
        $shippings_conditions_table = $schema->createTable('pongho_shippings_conditions');
        $shippings_conditions_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $shippings_conditions_table->addColumn('shipping_id', 'integer', ['unsigned' => true]);
        $shippings_conditions_table->addColumn('condition_by_weight', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $shippings_conditions_table->addColumn('condition_by_price', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $shippings_conditions_table->addColumn('cost', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $shippings_conditions_table->addColumn('position', 'integer', ['unsigned' => true, 'default' => 0]);

        $shippings_conditions_table->setPrimaryKey(['id']);
        $shippings_conditions_table->addForeignKeyConstraint($shippings_table, ['shipping_id'], ['id']);

        // Tipi di pagamento
        $payments_table = $schema->createTable('pongho_payments');
        $payments_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $payments_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
        $payments_table->addColumn('checkout_description', 'text', ['length' => 65535, 'default' => '']);
        $payments_table->addColumn('email_description', 'text', ['length' => 65535, 'default' => '']);
        $payments_table->addColumn('cost', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $payments_table->addColumn('handler_class', 'string', ['length' => 200, 'default' => '']);
        $payments_table->addColumn('handler_options', 'text', ['length' => 65535, 'default' => '']);
        $payments_table->addColumn('position', 'integer', ['unsigned' => true, 'default' => 0]);
        $payments_table->addColumn('is_enabled', 'boolean', ['default' => false]);

        $payments_table->setPrimaryKey(['id']);

        // Ordini
        $orders_table = $schema->createTable('pongho_orders');
        $orders_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $orders_table->addColumn('site_id', 'integer', ['unsigned' => true]);
        $orders_table->addColumn('session_id', 'string', ['length' => 50, 'default' => '']);
        $orders_table->addColumn('customer_id', 'integer', ['unsigned' => true]);
        $orders_table->addColumn('customer_ip', 'string', ['length' => 50, 'default' => '']);
        $orders_table->addColumn('status', 'string', ['length' => 50, 'default' => '']);
        $orders_table->addColumn('products_total', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_table->addColumn('products_count', 'integer', ['unsigned' => true, 'default' => 0]);
        $orders_table->addColumn('subtotal', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_table->addColumn('payment_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_table->addColumn('payment_cost', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_table->addColumn('payment_hash', 'string', ['length' => 32, 'default' => '']);
        $orders_table->addColumn('payment_details', 'text', ['length' => 65535, 'default' => '']);
        $orders_table->addColumn('shipping_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_table->addColumn('shipping_cost', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_table->addColumn('taxable', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_table->addColumn('tax_cost', 'decimal', ['precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_table->addColumn('total', 'decimal', ['precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_table->addColumn('invoice_name', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_surname', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_address1', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_address2', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_city', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_province', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_postcode', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_country_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_table->addColumn('invoice_telephone', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('invoice_address_text', 'text', ['length' => 65535, 'default' => '']);
        $orders_table->addColumn('invoice_code', 'string', ['length' => 50, 'default' => '']);
        $orders_table->addColumn('customer_code_name', 'string', ['length' => 50, 'default' => '']);
        $orders_table->addColumn('customer_code_value', 'string', ['length' => 50, 'default' => '']);
        $orders_table->addColumn('shipping_address_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_table->addColumn('shipping_name', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_surname', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_address1', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_address2', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_city', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_province', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_postcode', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_country_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_table->addColumn('shipping_telephone', 'string', ['length' => 100, 'default' => '']);
        $orders_table->addColumn('shipping_address_text', 'text', ['length' => 65535, 'default' => '']);
        $orders_table->addColumn('note', 'text', ['length' => 65535, 'default' => '']);
        $orders_table->addColumn('tracking_code', 'string', ['length' => 200, 'default' => '']);
        $orders_table->addColumn('process_note', 'text', ['length' => 65535, 'default' => '']);
        $orders_table->addColumn('notify_sent', 'boolean', ['default' => false]);
        $orders_table->addColumn('created_at', 'datetime');
        $orders_table->addColumn('updated_at', 'datetime');
        $orders_table->addColumn('ordered_at', 'datetime', ['notnull' => false]);
        $orders_table->addColumn('paid_at', 'datetime', ['notnull' => false]);
        $orders_table->addColumn('processed_at', 'datetime', ['notnull' => false]);
        $orders_table->addColumn('customer_is_guest', 'boolean', ['default' => false]);
        $orders_table->addColumn('customer_email', 'string', ['length' => 255, 'default' => '']);
        $orders_table->addColumn('invoice_province_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_table->addColumn('shipping_province_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_table->addColumn('payment_fail_notify_sent', 'boolean', ['default' => false]);
        $orders_table->addColumn('shopping_points', 'integer', ['notnull' => false, 'default' => 0, 'unsigned' => true]);
        $orders_table->addColumn('used_shopping_points', 'integer', ['notnull' => false, 'default' => 0, 'unsigned' => true]);

        $orders_table->setPrimaryKey(['id']);
        $orders_table->addIndex(['session_id']);
        $orders_table->addForeignKeyConstraint($schema->getTable('pongho_sites'), ['site_id'], ['id']);
        $orders_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['customer_id'], ['id']);
        $orders_table->addForeignKeyConstraint($payments_table, ['payment_id'], ['id']);
        $orders_table->addForeignKeyConstraint($shippings_table, ['shipping_id'], ['id']);
        $orders_table->addForeignKeyConstraint($schema->getTable('pongho_countries'), ['invoice_country_id'], ['id']);
        $orders_table->addForeignKeyConstraint($schema->getTable('pongho_addresses'), ['shipping_address_id'], ['id'], ['onDelete' => 'SET NULL']);
        $orders_table->addForeignKeyConstraint($schema->getTable('pongho_countries'), ['shipping_country_id'], ['id']);
        $orders_table->addForeignKeyConstraint($schema->getTable('pongho_provinces'), ['invoice_province_id'], ['id'], ['onDelete' => 'SET NULL']);
        $orders_table->addForeignKeyConstraint($schema->getTable('pongho_provinces'), ['shipping_province_id'], ['id'], ['onDelete' => 'SET NULL']);

        // Dettaglio degli ordini
        $orders_rows_table = $schema->createTable('pongho_orders_rows');
        $orders_rows_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $orders_rows_table->addColumn('order_id', 'integer', ['unsigned' => true]);
        $orders_rows_table->addColumn('product_id', 'integer', ['unsigned' => true]);
        $orders_rows_table->addColumn('size_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_rows_table->addColumn('row_id', 'integer', ['unsigned' => true]);
        $orders_rows_table->addColumn('quantity', 'integer', ['unsigned' => true, 'default' => 0]);
        $orders_rows_table->addColumn('price', 'decimal', ['precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_rows_table->addColumn('offer', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_rows_table->addColumn('discount', 'decimal', ['unsigned' => true, 'precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_rows_table->addColumn('discount_type', 'string', ['length' => 1, 'default' => '']);
        $orders_rows_table->addColumn('total', 'decimal', ['precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_rows_table->addColumn('shopping_points', 'integer', ['unsigned' => true, 'default' => 0]);
        $orders_rows_table->addColumn('is_previous_cart', 'boolean', ['default' => false]);
        $orders_rows_table->addColumn('is_unavailable', 'boolean', ['default' => false]);
        $orders_rows_table->addColumn('created_at', 'datetime');
        $orders_rows_table->addColumn('updated_at', 'datetime');

        $orders_rows_table->setPrimaryKey(['id']);
        $orders_rows_table->addUniqueIndex(['order_id', 'product_id', 'size_id']);
        $orders_rows_table->addForeignKeyConstraint($orders_table, ['order_id'], ['id']);
        $orders_rows_table->addForeignKeyConstraint($schema->getTable('pongho_nodes'), ['product_id'], ['id']);
        $orders_rows_table->addForeignKeyConstraint($schema->getTable('pongho_nodes_sizes'), ['size_id'], ['id']);

        $users_table = $schema->getTable('pongho_users');
        $users_table->addColumn('shipping_address_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $users_table->addColumn('shopping_points', 'integer', ['notnull' => false, 'default' => 0, 'unsigned' => true]);
        $users_table->addForeignKeyConstraint($schema->getTable('pongho_addresses'), ['shipping_address_id'], ['id']);

        // Sconti
        $order_discounts_table = $schema->createTable('pongho_shop_order_discounts');
        $order_discounts_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $order_discounts_table->addColumn('handler_name', 'string', ['length' => 255, 'default' => '']);
        $order_discounts_table->addColumn('code', 'string', ['length' => 255, 'default' => '']);
        $order_discounts_table->addColumn('description', 'string', ['length' => 255, 'default' => '']);
        $order_discounts_table->addColumn('settings', 'json');
        $order_discounts_table->addColumn('email_sender_name', 'string', ['length' => 255, 'default' => '']);
        $order_discounts_table->addColumn('email_sender_email', 'string', ['length' => 255, 'default' => '']);
        $order_discounts_table->addColumn('email_subject', 'string', ['length' => 255, 'default' => '']);
        $order_discounts_table->addColumn('email_body', 'text', ['length' => 65535, 'default' => '']);
        $order_discounts_table->addColumn('email_sent', 'boolean', ['default' => false]);
        $order_discounts_table->addColumn('valid_from', 'datetime', ['notnull' => false, 'default' => null]);
        $order_discounts_table->addColumn('expire_on', 'datetime', ['notnull' => false, 'default' => null]);
        $order_discounts_table->setPrimaryKey(['id']);

        // Condizioni degli sconti
        $order_discounts_conditions_table = $schema->createTable('pongho_shop_order_discounts_conditions');
        $order_discounts_conditions_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $order_discounts_conditions_table->addColumn('discount_id', 'integer', ['unsigned' => true, 'notnull' => true]);
        $order_discounts_conditions_table->addColumn('value', 'decimal', ['precision' => 12, 'scale' => 4, 'default' => 0]);
        $order_discounts_conditions_table->addColumn('value_type', 'string', ['length' => 1, 'default' => '']);
        $order_discounts_conditions_table->addColumn('condition', 'decimal', ['precision' => 12, 'scale' => 4, 'default' => 0]);
        $order_discounts_conditions_table->setPrimaryKey(['id']);
        $order_discounts_conditions_table->addForeignKeyConstraint($order_discounts_table, ['discount_id'], ['id']);

        // Relazione ordini / sconti
        $orders_order_discounts_table = $schema->createTable('pongho_shop_orders_order_discounts');
        $orders_order_discounts_table->addColumn('order_id', 'integer', ['unsigned' => true, 'notnull' => true]);
        $orders_order_discounts_table->addColumn('discount_id', 'integer', ['unsigned' => true, 'notnull' => true]);
        $orders_order_discounts_table->addColumn('name', 'string', ['length' => 255, 'default' => '']);
        $orders_order_discounts_table->addColumn('value', 'decimal', ['precision' => 12, 'scale' => 4, 'default' => 0]);
        $orders_order_discounts_table->addColumn('priority', 'integer', ['unsigned' => true, 'notnull' => false]);
        $orders_order_discounts_table->addForeignKeyConstraint($orders_table, ['order_id'], ['id']);
        $orders_order_discounts_table->addForeignKeyConstraint($order_discounts_table, ['discount_id'], ['id']);

        $this->updateSchema();

        $module = $this->getOrCreateModule(Kernel::FRONT_CONTROLLER);
        $this->getOrCreateModuleSite($module->id);
        $this->getOrCreateLanguageModule($module->id, '/shop/', 'Shop');

        /**
         * Permessi
         *
         * @var Role $admin_role
         */
        $admin_role = Role::find(Role::ADMIN);

        $admin_role
            ->addPermit('shop.admin')
            ->addPermit('shop.admin.orders')
            ->addPermit('shop.admin.payments.manage')
            ->addPermit('shop.admin.shippings.manage')
            ->addPermit('shop.buy')
            ->addPermit('shop.view_orders')
        ;

        Permit::findOrCreate('shop.admin.payments.add', false);
        Permit::findOrCreate('shop.admin.payments.delete', false);
        Permit::findOrCreate('shop.admin.shippings.add', false);
        Permit::findOrCreate('shop.admin.shippings.delete', false);
        Permit::findOrCreate('shop.admin.stats', false);
        Permit::findOrCreate('shop.admin.stats.sold', false);
        Permit::findOrCreate('shop.admin.stats.ordered', false);

        // TODO: FS#454 - Permessi taglie
        $admin_role->addPermit('shop.admin.sizes');

        /**
         * Traduzioni
         */
        $this->installTranslationsFromFile(__DIR__ . '/Resources/translations.json');
    }

    /**
     * Ripulisce il database e ripristina quanto è possibile allo stato precedente all'installazione
     */
    public function uninstall()
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->container->get('logger');

        $logger->info('[Showcase][Shop][Installer] Pulizia del database');

        // Rimuovo le tabelle dello Showcase
        $schema = $this->getSchema();

        if ($schema->hasTable('pongho_shop_orders_order_discounts')) {
            $schema->dropTable('pongho_shop_orders_order_discounts');
        }

        if ($schema->hasTable('pongho_shop_order_discounts_conditions')) {
            $schema->dropTable('pongho_shop_order_discounts_conditions');
        }

        if ($schema->hasTable('pongho_shop_order_discounts')) {
            $schema->dropTable('pongho_shop_order_discounts');
        }

        if ($schema->hasTable('pongho_orders_rows')) {
            $schema->dropTable('pongho_orders_rows');
        }

        if ($schema->hasTable('pongho_orders')) {
            $schema->dropTable('pongho_orders');
        }

        if ($schema->hasTable('pongho_orders_discounts_conditions')) {
            $schema->dropTable('pongho_orders_discounts_conditions');
        }

        if ($schema->hasTable('pongho_orders_discounts')) {
            $schema->dropTable('pongho_orders_discounts');
        }

        if ($schema->hasTable('pongho_payments')) {
            $schema->dropTable('pongho_payments');
        }

        if ($schema->hasTable('pongho_shippings_conditions')) {
            $schema->dropTable('pongho_shippings_conditions');
        }

        if ($schema->hasTable('pongho_shippings')) {
            $schema->dropTable('pongho_shippings');
        }

        $users_table = $schema->getTable('pongho_users');

        if ($users_table->hasColumn('shipping_address_id')) {
            foreach ($users_table->getForeignKeys() as $fk) {
                if ($fk->getColumns() === ['shipping_address_id']) {
                    $users_table->removeForeignKey($fk->getName());
                }
            }

            $users_table->dropColumn('shipping_address_id');
        }

        if ($users_table->hasColumn('shopping_points')) {
            $users_table->dropColumn('shopping_points');
        }

        $this->updateSchema();

        // Rimuovo le impostazioni relative allo Showcase dal sito e dai tipi di nodo

        /** @var \Application\Cms\Model\Manager\NodeTypeManager $node_type_manager */
        $node_type_manager = $this->container->get('node_type_manager');

        /** @var \Application\Core\Model\LanguageSite $language_site */
        $language_site = $this->container->get('language_site');

        foreach ($node_type_manager->getNodeTypes($language_site->language_id) as $node_type) {
            $options = [
                'enable_code',
                'enable_price',
                'enable_availabilities',
                'hide_unavailable_from_archives',
                'enable_sizes',
            ];

            foreach ($options as $option) {
                $node_type->delOption($option);
            }

            $node_type->save();
        }

        $this->deleteModule(Kernel::FRONT_CONTROLLER);

        // Permessi
        foreach (Permit::findByPrefix('shop.') as $permit) {
            $permit->delete();
        }
    }
}
