<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\GoogleShopping\Form\FieldView;

use Pongho\Form\Field\TextField;
use Pongho\Sdk\GoogleShopping\Values\GoogleProductCategories;
use Pongho\Sdk\GoogleShopping\Values\GoogleProductCategoriesFactory;
use Pongho\Template\HeadHelper;

class GoogleCategoryFieldView
{
    /**
     * @var HeadHelper
     */
    protected $head;

    public function __construct(HeadHelper $head)
    {
        $this->head = $head;
    }

    /**
     * @return string
     */
    public function __invoke(TextField $field)
    {
        $provider = (new GoogleProductCategoriesFactory())->create();

        $this->head->addJavaScriptInline($this->renderJS($field, $provider));

        return $this->renderHTML($field);
    }

    /**
     * @return string
     */
    protected function renderJS(TextField $field, GoogleProductCategories $provider)
    {
        return <<<JS
$(function () {
    var app = {
        gCats: {$provider->toJson()},
        init: function () {
            app.field = $("#{$field->getId()}");
            app.value = app.field.val();
            app.selects = $('<div />').insertAfter(app.field);

            if (app.value) {
                app.buildSelects(app.value);
            } else {
                app.prepare();
            }
        },
        buildSelects: function (category) {
            var path = app.toPath(category),
                subPath = path[0];

            app.buildSelect("").val(path[0]);

            path.forEach(function (segment, index) {
                var next = path[index + 1],
                    select = app.buildSelect(subPath);

                if (next) {
                    select.val(next);
                    subPath += ' > ' + next;
                }
            });
        },
        prepare: function () {
            app.buildSelect("");
        },
        toPath: function (category) {
            return category.split('>').map(function (str) {
                return str.trim();
            });
        },
        toCategory: function (path) {
            return path.join(' > ');
        },
        buildSelect: function (parent, selected) {
            if (!app.gCats[parent]) {
                return;
            }

            return $("<select />")
                .css({
                    display: "block",
                    marginBottom: "2px"
                })
                .html(function () {
                    var options = '<option></option>';

                    (app.gCats[parent] || []).forEach(function (cat) {
                        options += '<option' + (cat === selected ? ' selected' : '') + '>' + cat + '</option>';
                    });

                    return options;
                })
                .on('change', function () {
                    var select = $(this),
                        path = [],
                        category;

                    select.nextAll('select').remove();

                    app.selects.find('select').each(function () {
                        var value = $(this).val();

                        if (value) {
                            path.push(value);
                        }
                    });

                    category = app.toCategory(path);
                    app.field.val(category);

                    if (select.val() && app.gCats[category]) {
                        app.buildSelect(category);
                    }
                })
                .appendTo(app.selects);
        }
    };

    app.init();

    app.field.closest('.form-wrapper').on('after-update', app.init);
});
JS;
    }

    /**
     * @return string
     */
    protected function renderHTML(TextField $field)
    {
        return <<<HTML
<div class="control-group">
    <label class="control-label">{$field->getLabel()}</label>
    <div class="controls">{$field->getHiddenField()}</div>
</div>
HTML;
    }
}
