<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Form\Builder;

use Application\Core\Model\Application;
use Application\Core\Model\Manager\ApplicationManagerInterface;
use Application\Core\Model\Province;
use Application\Core\Model\Site;
use Application\Showcase\Form\Subject\LoginSubject;
use Pongho\Core\LocalizationInterface;
use Pongho\Form\Field\CheckboxField;
use Pongho\Form\Field\HiddenField;
use Pongho\Form\Field\PasswordField;
use Pongho\Form\Field\RadioField;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Field\TextField;
use Pongho\Form\Fieldset;
use Pongho\Form\Form;
use Pongho\Form\Utilities\FormBuilder;
use Pongho\Form\Utilities\FormConfig;

class LoginFormBuilder
{
    /**
     * @var LocalizationInterface
     */
    protected $lang;

    /**
     * @var LoginSubject
     */
    protected $subject;

    /**
     * @var \Application\Core\Model\Site
     */
    protected $site;

    /**
     * @var ApplicationManagerInterface
     */
    protected $app_manager;

    public function __construct(
        LoginSubject $subject,
        LocalizationInterface $lang,
        Site $site,
        ApplicationManagerInterface $app_manager
    ) {
        $this->subject = $subject;
        $this->lang = $lang;
        $this->site = $site;
        $this->app_manager = $app_manager;
    }

    /**
     * @return Form
     */
    public function build()
    {
        $config = new FormConfig('sign-in', $this->subject);

        $config->addField(
            'username_or_email',
            [
                'class'      => TextField::class,
                'label'      => $this->lang->get('email'),
                'attributes' => [
                    'maxlength'    => 50,
                    'autocomplete' => 'off',
                ],
            ]
        );

        $config->addField(
            'type',
            [
                'class'   => RadioField::class,
                'options' => [
                    LoginSubject::TYPE_SUBSCRIBE => $this->lang->get('shop_subscribe'),
                    LoginSubject::TYPE_LOGIN     => $this->lang->get('shop_login'),
                ],
            ]
        );

        $this->buildLoginFieldset($config);
        $this->buildSubscribeFieldset($config);

        return FormBuilder::buildConfig($config);
    }

    protected function buildLoginFieldset(FormConfig $config)
    {
        $config->addField(
            'login',
            [
                'class' => Fieldset::class,
            ]
        );

        $config->addField(
            'login/login_password',
            [
                'class'      => PasswordField::class,
                'label'      => $this->lang->get('password'),
                'attributes' => [
                    'maxlength' => 50,
                ],
            ]
        );
    }

    protected function buildSubscribeFieldset(FormConfig $config)
    {
        $config->addField(
            'subscribe',
            [
                'class' => Fieldset::class,
                'label' => $this->lang->get('shop_subscribe'),
            ]
        );

        $this->buildShippingAddressFieldset($config);
        $this->buildInvoiceInfoFieldset($config);
        $this->buildAccountFieldset($config);
    }

    /**
     * buildSubscribeFieldset
     */
    protected function buildAccountFieldset(FormConfig $config)
    {
        $config->addField(
            'subscribe/account',
            [
                'class' => Fieldset::class,
                'label' => $this->lang->get('account'),
            ]
        );

        if ($this->site->getOption('enable_guest_checkout')) {
            $config->addField(
                'subscribe/account/checkout_mode',
                [
                    'class'   => RadioField::class,
                    'options' => [
                        LoginSubject::CHECKOUT_GUEST  => $this->lang->get('shop_checkout_guest'),
                        LoginSubject::CHECKOUT_NORMAL => $this->lang->get('shop_checkout_normal'),
                    ],
                ]
            );
        }

        $config->addFields(
            'subscribe/account',
            [
                'account_password'         => [
                    'class'      => PasswordField::class,
                    'label'      => $this->lang->get('password'),
                    'attributes' => [
                        'maxlength'    => 50,
                        'autocomplete' => 'new-password',
                    ],
                ],
                'account_confirm_password' => [
                    'class'      => PasswordField::class,
                    'label'      => $this->lang->get('confirm_password'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
            ]
        );

        if ($this->app_manager->isEnabled('Newsletter', $this->site)) {
            $config->addField(
                'subscribe/account/newsletter',
                [
                    'class' => CheckboxField::class,
                    'label' => $this->lang->get('newsletter'),
                ]
            );
        }
    }

    protected function buildShippingAddressFieldset(FormConfig $config)
    {
        $config->addField(
            'subscribe/shipping_address',
            [
                'class' => Fieldset::class,
                'label' => $this->lang->get('shipping_address'),
            ]
        );

        $this->buildAddressFieldset($config, 'subscribe/shipping_address', 'shipping_address_');
    }

    /**
     * buildShippingAddressFieldset
     */
    protected function buildInvoiceInfoFieldset(FormConfig $config)
    {
        $config->addField(
            'subscribe/invoice_info',
            [
                'class' => Fieldset::class,
                'label' => $this->lang->get('invoice_info'),
            ]
        );

        $config->addField(
            'subscribe/invoice_info/invoice_address_mode',
            [
                'class'   => RadioField::class,
                'options' => [
                    LoginSubject::INVOICE_ADDRESS_SAME      => $this->lang->get('shop_invoice_same_address_option'),
                    LoginSubject::INVOICE_ADDRESS_DIFFERENT => $this->lang->get(
                        'shop_invoice_different_address_option'
                    ),
                ],
            ]
        );

        $config->addField(
            'subscribe/invoice_info/address',
            [
                'class' => Fieldset::class,
            ]
        );

        $this->buildAddressFieldset($config, 'subscribe/invoice_info/address', 'invoice_info_');

        $config->addField(
            'subscribe/invoice_info/customer_code',
            [
                'class'      => TextField::class,
                'label'      => $this->lang->get('codice_fiscale'), // @todo I18n
                'attributes' => [
                    'maxlength' => 50,
                ],
            ]
        );
    }

    /**
     * @param            $path
     * @param string     $prefix
     */
    protected function buildAddressFieldset(FormConfig $config, $path, $prefix)
    {
        $config->addFields(
            $path,
            [
                $prefix . 'name'        => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('name'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                $prefix . 'surname'     => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('surname'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                $prefix . 'address1'    => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('address'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                $prefix . 'address2'    => [
                    'class'       => TextField::class,
                    'label'       => $this->lang->get('address_info'),
                    'description' => $this->lang->get('address_info_help'),
                    'attributes'  => [
                        'maxlength' => 50,
                    ],
                ],
                $prefix . 'postcode'    => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('postcode'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                $prefix . 'city'        => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('city'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
                $prefix . 'province_id' => [
                    'class'    => SelectField::class,
                    'label'    => $this->lang->get('province'),
                    'options'  => $this->getProvinces(),
                    'settings' => [
                        'placeholder' => $this->lang->get('select_province'),
                    ],
                ],
                $prefix . 'country_id'  => [
                    'class' => HiddenField::class,
                    'label' => $this->lang->get('country'),
                ],
                $prefix . 'telephone'   => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('telephone'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
            ]
        );
    }

    protected function buildCustomerCodeFieldset(FormConfig $config)
    {
        $config->addField(
            'customer-code',
            [
                'class' => Fieldset::class,
            ]
        );

        // @todo I18n
        // @todo Possibilità di personalizzare
        // @todo Partita IVA?

        $config->addFields(
            'customer-code',
            [
                'customer_code' => [
                    'class'      => TextField::class,
                    'label'      => $this->lang->get('codice_fiscale'),
                    'attributes' => [
                        'maxlength' => 50,
                    ],
                ],
            ]
        );
    }

    /**
     * @return array
     *
     * @todo I18n
     */
    protected function getProvinces()
    {
        static $provinces;

        if ($provinces === null) {
            $options = ['order' => 'name'];

            /** @var Province $province */
            $provinces = [];
            foreach (Province::all($options) as $province) {
                $provinces[$province->id] = $province->name;
            }
        }

        return $provinces;
    }
}
