<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Entity;

use Application\Showcase\Exception\OrderException;
use Application\Showcase\Model\Node as Product;
use Application\Showcase\Model\Size;
use Application\Showcase\ShoppingPoints\Scheme;
use Application\Showcase\Utilities\Taxation;

/**
 * OrderInterface
 *
 * TODO Completare.
 */
interface OrderInterface
{
    /**
     * @return int
     */
    public function getId();

    /**
     * Restituisce le righe dell’ordine.
     *
     * @param bool $force_update Se forzare il caricamento delle righe dal database, invece di usare la cache interna.
     * @return OrderRowInterface[]
     */
    public function getRows($force_update = false);

    /**
     * Aggiunge un prodotto.
     *
     * @param int     $quantity
     * @param Size    $size
     * @return bool
     * @throws OrderException Se ci sono problemi (generalmente di disponibilità).
     *
     * TODO: ProductInterface e SizeInterface.
     */
    public function addProduct(Product $product, $quantity = 1, ?Size $size = null);

    /**
     * Restituisce il costo totale dei prodotti.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function productsTotal(?Taxation $taxation = null);

    /**
     * Restituisce il costo totale formattato dei prodotti.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatProductsTotal(?Taxation $taxation = null);

    /**
     * Restituisce il subtotale.
     *
     * Il subtotale è il costo totale dei prodotti sottratto di un eventuale buono sconto.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function subtotal(?Taxation $taxation = null);

    /**
     * Restituisce il subtotale formattato.
     *
     * Il subtotale è il costo totale dei prodotti sottratto di un eventuale buono sconto.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatSubtotal(?Taxation $taxation = null);

    /**
     * Restituisce i costi di pagamento.
     *
     * Questo metodo è pensato per il template.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function paymentCost(?Taxation $taxation = null);

    /**
     * Restituisce i costi di pagamento formattati.
     *
     * Questo metodo è pensato per il template.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatPaymentCost(?Taxation $taxation = null);

    /**
     * Restituisce i costi di spedizione.
     *
     * Questo metodo è pensato per il template.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return float
     */
    public function shippingCost(?Taxation $taxation = null);

    /**
     * Restituisce i costi di spedizione formattati.
     *
     * Questo metodo è pensato per il template.
     *
     * @param Taxation $taxation Oggetto per il calcolo delle tasse.
     * @return string
     */
    public function formatShippingCost(?Taxation $taxation = null);

    /**
     * Restituisce l’imponibile.
     *
     * @return float
     */
    public function taxable();

    /**
     * Restituisce l’imponibile formattato.
     *
     * @return string
     */
    public function formatTaxable();

    /**
     * Restituisce il costo delle tasse.
     *
     * @return float
     */
    public function taxCost();

    /**
     * Restituisce il costo formattato delle tasse.
     *
     * @return string
     */
    public function formatTaxCost();

    /**
     * Restituisce il totale.
     *
     * @return float
     */
    public function total();

    /**
     * Restituisce il totale formattato.
     *
     * @return string
     */
    public function formatTotal();

    /**
     * Restituisce il numero totale di prodotti in carrello / ordine.
     *
     * @return int
     */
    public function quantities();

    /**
     * Indica se l’ordine è stato fatto da un utente ospite.
     *
     * @return bool
     */
    public function customerIsGuest();

    /**
     * Indica se il cliente associato all’ordine si è loggato.
     *
     * @return bool
     */
    public function customerIsLogged();

    /**
     * Restituisce l’indirizzo e-mail del cliente.
     *
     * @return string
     */
    public function getCustomerEmail();

    /**
     * Restituisce il nominativo del cliente.
     *
     * @return string
     */
    public function getCustomerName();

    /**
     * Restituisce il nome del cliente.
     *
     * @return string
     */
    public function getCustomerFirstName();

    /**
     * Restituisce il cognome del cliente.
     *
     * @return string
     */
    public function getCustomerLastName();

    /**
     * Restituisce i punti accumulati dal cliente contando anche l’ordine corrente.
     *
     * @return int
     */
    public function getGainedCustomerShoppingPoints();

    /**
     * Restituisce i punti accumulati dal cliente e subito utilizzabili.
     *
     * @return int
     */
    public function getAvailableCustomerShoppingPoints();

    /**
     * Imposta i punti della raccolta punti.
     *
     * @param int                    $used_points
     * @return $this
     * @throws \Application\Showcase\ShoppingPoints\Exception\ShoppingPointsException Se non è possibile impostare i punti.
     */
    public function setUsedShoppingPoints($used_points, Scheme $scheme);

    /**
     * @return int
     */
    public function getShoppingPoints();
}
