<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Domain\Order\Command;

use Application\Core\Mailer\Helper;
use Application\Core\Model\Site;
use Application\Showcase\Model\Order;
use Pongho\Core\LocalizationInterface;
use Pongho\Template\View;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;
use Symfony\Component\Mime\Email;
use Symfony\Component\Mime\Part\DataPart;

class ProcessOrderNotifier
{
    /**
     * @var MailerInterface
     */
    protected $mailer;

    /**
     * @var Helper
     */
    private $mailerHelper;

    /**
     * @var Site
     */
    protected $site;

    /**
     * @var View
     */
    protected $view;

    /**
     * @var LocalizationInterface
     */
    protected $lang;

    public function __construct(
        MailerInterface $mailer,
        Helper $mailerHelper,
        Site $site,
        View $view,
        LocalizationInterface $lang
    ) {
        $this->mailer = $mailer;
        $this->mailerHelper = $mailerHelper;
        $this->site = $site;
        $this->view = $view;
        $this->lang = $lang;
    }

    /**
     * @param DataPart[] $attachments
     * @throws TransportExceptionInterface
     *
     * @customized HDC
     */
    public function send(Order $order, array $attachments = [])
    {
        $companyName = $this->site->getOption('company_name');
        $companyEmail = $this->site->getOption('company_orders_email', $this->site->getOption('company_email'));

        $customerName = $order->getCustomerName();
        $customerEmail = $order->getCustomerEmail();

        $this->view->assignVars([
            'order'          => $order,
            'order_user_url' => $this->site->getPermalink() . 'shop/order/' . $order->id . '/',
        ]);

        $email = (new Email())
            ->from(new Address($companyEmail, $companyName))
            ->to(new Address($customerEmail, $customerName))
            ->subject(
                sprintf(
                    $this->lang->get('email_shop_order_processed_subject'),
                    $this->site->getName(),
                    $order->id
                )
            )
            ->html(
                $this->mailerHelper->replaceVars(
                    file_get_contents($this->site->getThemesPath('email/base.php')),
                    [
                        'CONTENT' => $this->view->render(),
                    ]
                )
            );

        foreach ($attachments as $attachment) {
            $email->addPart($attachment);
        }

        $this->mailer->send($this->mailerHelper->prepareEmail($email));
    }
}
