<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Domain\Order\Command;

use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Discount\OrderDiscounts;
use Application\Showcase\Domain\CommandInterface;
use Application\Showcase\Domain\CommandResult;
use Application\Showcase\Model\Order;
use Pongho\Utilities\DateTime;

/**
 * @todo Disponibilità sul nodo
 * @todo Tracking code
 */
class ProcessOrderCommand implements CommandInterface
{
    /**
     * @var OrderDiscounts
     */
    protected $orderDiscounts;

    /**
     * @var ProcessOrderNotifier
     */
    protected $notificationSender;

    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var Order
     */
    protected $order;

    /**
     * @var DateTime
     */
    protected $date;

    /**
     * @var string
     */
    protected $trackingCode;

    /**
     * @var string
     */
    protected $note;

    /**
     * @var bool
     */
    protected $hasNotAvailableRows = false;

    /**
     * @var bool
     */
    protected $successful = true;

    /**
     * @var array
     */
    protected $alerts = [];

    /**
     * @var array
     */
    protected $errors = [];

    public function __construct(
        OrderDiscounts $orderDiscounts,
        ProcessOrderNotifier $sender,
        Translator $translator
    ) {
        $this->orderDiscounts = $orderDiscounts;
        $this->notificationSender = $sender;
        $this->translator = $translator;
    }

    /**
     * Il reset deve essere esguito prima di processare un’altro ordine.
     *
     * @return $this
     *
     * @customized HDC
     */
    public function reset()
    {
        $this->order = null;
        $this->date = null;
        $this->trackingCode = null;
        $this->note = null;

        return $this;
    }

    /**
     * @param Order $order
     * @return $this
     */
    public function setOrder($order)
    {
        $this->order = $order;

        return $this;
    }

    /**
     * @return $this
     */
    public function setDate(DateTime $date)
    {
        $this->date = $date;

        return $this;
    }

    /**
     * @param string $trackingCode
     * @return $this
     */
    public function setTrackingCode($trackingCode = null)
    {
        if (null !== $trackingCode) {
            $this->trackingCode = trim($trackingCode);
        }

        return $this;
    }

    /**
     * @param string $note
     * @return $this
     */
    public function setNote($note = null)
    {
        if (null !== $note) {
            $this->note = trim($note);
        }

        return $this;
    }

    public function handle()
    {
        if (null === $this->order) {
            throw new \LogicException('The order must be set before handle the command.');
        }

        if ($this->order->isProcessed()) {
            return new CommandResult(true);
        }

        // reset result
        $this->successful = true;
        $this->alerts = [];
        $this->errors = [];

        // handle
        try {
            $this->handleRaw();
            $this->persist();
            $this->sendNotification();
        } catch (\Exception $exception) {
            $this->successful = false;
            $this->errors[] = $exception->getMessage();
        }

        $message = '';
        if ($this->successful) {
            $message = $this->translator->trans('The order has been processed.');
        }

        return new CommandResult(
            $this->successful,
            $message,
            $this->errors,
            $this->alerts
        );
    }

    /**
     * @throws \Exception
     */
    protected function handleRaw()
    {
        if (null === $this->date) {
            $this->date = new DateTime();
        }

        $this->order->setProcessed($this->date);

        if (null !== $this->trackingCode) {
            $this->order->setTrackingCode($this->trackingCode);
        }

        if (null !== $this->note) {
            $this->order->setProcessNote($this->note);
        }

        $this->handleShoppingPoints();
    }

    protected function handleShoppingPoints()
    {
        if ($this->orderDiscounts->isEnabled('shopping_points')) {
            $customer = $this->order->customer;

            if (
                $this->order->used_shopping_points > 0
                && $this->order->getAvailableCustomerShoppingPoints() >= $this->order->used_shopping_points
            ) {
                $customer->shopping_points -= $this->order->used_shopping_points;
            } elseif ($this->autoEarnShoppingPoints()) {
                $customer->shopping_points += $this->order->shopping_points;
            }
        }
    }

    /**
     * Indica se i punti possono essere incrementati in automatico all’evasione dell’ordine.
     *
     * Questo controllo può essere personalizzato nel caso i punti vengano gestiti tramite altro
     * sistema, come ad esempio le API di HDC.
     *
     * @return bool
     */
    protected function autoEarnShoppingPoints()
    {
        return true;
    }

    /**
     * @throws \Exception
     */
    protected function persist()
    {
        if (!$this->successful) {
            return;
        }

        if (!Order::transaction($this->transaction())) {
            $this->successful = false;
            $this->errors[] = $this->translator->trans('Could not update the order!');

            return;
        }

        if ($this->hasNotAvailableRows) {
            $this->alerts[] = $this->translator->trans('The order has unavailable rows');
        }
    }

    /**
     * @return \Closure
     */
    protected function transaction()
    {
        return function () {
            if (!$this->order->save()) {
                return false;
            }

            if (!$this->order->customer->save(false)) {
                return false;
            }

            $this->order->updateBounds();

            /** @var \Application\Showcase\Model\OrderRow $row */
            foreach ($this->order->getRows() as $row) {
                if ($row->size_id) {
                    $size = $row->size();

                    if ($size->availabilities() < $row->quantity()) {
                        $this->hasNotAvailableRows = true;
                        $size->availabilities = 0;
                    } else {
                        $size->availabilities -= $row->quantity;
                    }

                    if (!$size->save()) {
                        return false;
                    }
                }

                $product = $row->product;
                $product->availabilities -= $row->quantity;
                $product->save();
            }

            return true;
        };
    }

    /**
     * @customized HDC
     */
    protected function sendNotification()
    {
        if (!$this->successful) {
            return;
        }

        $this->notificationSender->send($this->order);
    }
}
