<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Discount\Order;

use Application\Core\Localization;
use Application\Showcase\Model\Order;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\ShoppingPoints\Scheme;

class ShoppingPointsDiscount
{
    /**
     * @var Localization
     */
    private $lang;

    /**
     * @var Scheme
     */
    private $scheme;

    /**
     * @var OrderDiscount
     */
    private $discount;

    public function __construct(Localization $lang, Scheme $scheme)
    {
        $this->lang = $lang;
        $this->scheme = $scheme;
    }

    /**
     * @return Scheme
     */
    public function getScheme()
    {
        return $this->scheme;
    }

    /**
     * Aggiorna lo sconto dato dai punti utilizzati e restituisce il valore dello sconto.
     *
     * @param int   $usedPoints
     * @return float
     */
    public function update(Order $order, $usedPoints)
    {
        $order->setUsedShoppingPoints($usedPoints, $this->scheme);
        $discount = $this->getDiscount();

        if ($usedPoints) {
            $value = $this->getValue($usedPoints);
            $name = $this->getName($usedPoints);

            $order
                ->updateOrApplyDiscount($discount, $name, $value)
                ->save();

            return $value;
        }

        $order->removeDiscount($discount);

        return 0;
    }

    /**
     * @param int $usedPoints
     * @return string
     */
    protected function getName($usedPoints)
    {
        return sprintf($this->lang->get('shopping_points_discount'), $usedPoints);
    }

    /**
     * @param int $usedPoints
     * @return float
     */
    protected function getValue($usedPoints)
    {
        if ($usedPoints) {
            return $this->scheme->getPointValue() * $usedPoints;
        }

        return 0;
    }

    /**
     * @return OrderDiscount
     */
    protected function getDiscount()
    {
        if ($this->discount === null) {
            $this->discount = OrderDiscount::findByHandler('shopping_points');
        }

        return $this->discount;
    }
}
