<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\DependencyInjection;

use Application\Cms\Search\SearchableField;
use Application\Cms\Search\SearchableFields;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Mailer\Helper;
use Application\Core\Model\Site;
use Application\Showcase\Controller\Admin\Orders\EditPanel\CancelOrderPanel;
use Application\Showcase\Controller\Admin\Orders\EditPanel\ConfirmOrderPaymentPanel;
use Application\Showcase\Controller\Admin\Orders\EditPanel\ProcessOrderPanel;
use Application\Showcase\Discount\DiscountCalculator;
use Application\Showcase\Discount\Order\CouponCodeDiscount;
use Application\Showcase\Discount\Order\OrderAmountDiscount;
use Application\Showcase\Discount\Order\ShoppingPointsDiscount;
use Application\Showcase\Discount\OrderDiscounts;
use Application\Showcase\Discount\OrderDiscountsFacade;
use Application\Showcase\Domain\Order\Command\CancelOrderCommandFactory;
use Application\Showcase\Domain\Order\Command\ConfirmOrderPaymentCommandFactory;
use Application\Showcase\Domain\Order\Command\ProcessOrderCommandFactory;
use Application\Showcase\Domain\Order\Command\ProcessOrderNotifier;
use Application\Showcase\Export\DataExporter\OrdersExporter;
use Application\Showcase\Form\Builder\LoginFormBuilder;
use Application\Showcase\Form\Subject\LoginSubject;
use Application\Showcase\Installer\NodeTypeConverter;
use Application\Showcase\Model\Manager\ProductManager;
use Application\Showcase\Model\Manager\ProductManagerInterface;
use Application\Showcase\Model\Order;
use Application\Showcase\ShoppingPoints;
use Application\Showcase\Utilities\CheckoutBreadcrumbBuilder;
use Application\Showcase\Utilities\CheckoutOptions;
use Application\Showcase\Utilities\Config;
use Application\Showcase\Utilities\OrderNotification;
use Application\Showcase\Utilities\Taxation;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;
use Pongho\Template\View;
use Symfony\Component\Mailer\Mailer;

class ShowcaseServiceProvider implements ServiceProviderInterface
{
    public function register(Container $container): void
    {
        $container->setParameter('checkout_breadcrumb_builder_class', CheckoutBreadcrumbBuilder::class);
        $container->setParameter('shop_taxation_class', Taxation::class);
        $container->setParameter('shop_coupon_discount_class', CouponCodeDiscount::class);
        $container->setParameter('shop_order_amount_discount_class', OrderAmountDiscount::class);
        $container->setParameter('shop_login_form_subject_class', LoginSubject::class);
        $container->setParameter('shop_login_form_builder_class', LoginFormBuilder::class);

        $container->share('product_manager', function (): ProductManagerInterface {
            return new ProductManager();
        });

        $container->share('cart', function (Container $container): Order {
            // Ricavo il carrello standard...
            $cart = Order::getCart(
                $container->get('current_user'),
                $container->get('session'),
                $container->get('site'),
            );

            // ...e se non c’è lo creo.
            if ($cart == null) {
                $cart = Order::createCart(
                    $container->get('current_user'),
                    $container->get('session'),
                    $container->get('site'),
                    $container->get('request'),
                );
            }

            /** @var \Application\Showcase\Utilities\CheckoutOptions $checkout_options */
            $checkout_options = $container->get('shop_checkout_options');

            $cart->setCheckoutOptions($checkout_options);

            return $cart;
        });

        $container->share('checkout_breadcrumb', function (Container $container): CheckoutBreadcrumbBuilder {
            $class = $container->getParameter('checkout_breadcrumb_builder_class');

            return new $class(
                $container->get('language'),
                $container->get('request'),
            );
        });

        $container->share('shop_checkout_options', function (): CheckoutOptions {
            return new CheckoutOptions();
        });

        $container->share('shop_taxation', function (Container $container): Taxation {
            $class = $container->getParameter('shop_taxation_class');

            return new $class($container->get('site'));
        });

        $container->share('shop_login_form_subject', function (Container $container): LoginSubject {
            $class = $container->getParameter('shop_login_form_subject_class');

            return new $class(
                $container->get('cart'),
                $container->get('current_user'),
                $container->get('language'),
                $container->get(Mailer::class),
                $container->get(Helper::class),
                $container->get('site'),
                $container->get('application_manager'),
                $container->get('address_validator_factory'),
                $container->get('address_renderer_factory'),
            );
        });

        $container->share('shop_login_form_builder', function (Container $container): LoginFormBuilder {
            $class = $container->getParameter('shop_login_form_builder_class');

            return new $class(
                $container->get('shop_login_form_subject'),
                $container->get('language'),
                $container->get('site'),
                $container->get('application_manager'),
            );
        });

        $container->share('shop_order_notification', function (Container $container): OrderNotification {
            return new OrderNotification(
                $container->get('language'),
                $container->get(Mailer::class),
                $container->get(Helper::class),
                $container->get('global_view_vars'),
                $container->get('logger'),
            );
        });

        $container->share('showcase_node_type_converter', function (Container $container): NodeTypeConverter {
            return new NodeTypeConverter(
                $container->get('taxonomy_builder'),
            );
        });

        $container->share('admin.orders.exporter', function (Container $container): OrdersExporter {
            return new OrdersExporter(
                $container->get('connection'),
                $container->get('translator'),
                $container->get('language'),
            );
        });

        /**
         * Sconti
         */
        $container->share('shop_order_discounts', function (Container $container): OrderDiscounts {
            /** @var \Application\Core\Model\Site $site */
            $site = $container->get('site');

            return new OrderDiscounts(
                $site->getOption('order_discounts'),
            );
        });

        $container->share('shop_order_discounts_facade', function (Container $container): OrderDiscountsFacade {
            return new OrderDiscountsFacade(
                $container->get('shop_coupon_discount'),
                $container->get('shop_order_amount_discount'),
                $container->get('shop_shopping_points_discount'),
                $container->get('shop_order_discounts'),
                $container->get('language'),
                $container->get('logger'),
            );
        });

        $container->share('shop_coupon_discount', function (Container $container): CouponCodeDiscount {
            $class = $container->getParameter('shop_coupon_discount_class');

            return new $class(
                $container->get('language'),
                $container->get('current_user'),
                $container->get('node_type_manager'),
            );
        });

        $container->share('shop_order_amount_discount', function (Container $container): OrderAmountDiscount {
            $class = $container->getParameter('shop_order_amount_discount_class');

            return new $class(
                $container->get('language'),
            );
        });

        $container->share('shop_shopping_points_scheme', function (Container $container): ShoppingPoints\Scheme {
            /** @var \Application\Core\Model\Site $site */
            $site = $container->get('site');

            return new ShoppingPoints\Scheme(
                $site->getOption('shopping_point_cost'),
                $site->getOption('shopping_point_value'),
                $site->getOption('shopping_point_min_points'),
                $site->getOption('shopping_point_min_total_order'),
                $site->getOption('shopping_point_on_single_product'),
                $site->getOption('shopping_point_calculate_on_subtotal'),
            );
        });

        $container->share('shop_shopping_points_discount', function (Container $container): ShoppingPointsDiscount {
            return new ShoppingPointsDiscount(
                $container->get('language'),
                $container->get('shop_shopping_points_scheme'),
            );
        });

        $container->share('shop_discount_calculator', function (): DiscountCalculator {
            return new DiscountCalculator();
        });

        $container->extend('site', $this->siteExtension(...));

        $container->extend('main_site', $this->siteExtension(...));

        $container->extend(
            'translator',
            function (Translator $translator, Container $container): void {
                $translator->addResource('pongho_app', 'Showcase', $container->get('current_user')->language->culture);
            },
        );

        $container->extend(
            'global_view_vars',
            function (array $vars, Container $container): array {
                $vars['cart'] = $container->get('cart');
                $vars['shopping_points_scheme'] = $container->get('shop_shopping_points_scheme');
                $vars['order_discounts'] = $container->get('shop_order_discounts');

                return $vars;
            },
        );

        $container->extend(
            'search_searchable_fields',
            function (SearchableFields $fields): void {
                $fields->addField(new SearchableField('code'));
            },
        );

        /*
         * Admin / Orders
         */
        $container->share('shop.admin.orders.cancel_order_panel', function (Container $container): CancelOrderPanel {
            return new CancelOrderPanel(
                $container->get('translator'),
                $container->get('pongho_url'),
                $container->get('nodetype_helper'),
                $container->get('shop.domain.orders.commands.cancel_order_factory'),
                $container->get('logger'),
            );
        });

        $container->share(
            'shop.domain.orders.commands.cancel_order_factory',
            function (Container $container): CancelOrderCommandFactory {
                return new CancelOrderCommandFactory(
                    $container->get('translator'),
                );
            },
        );

        $container->share(
            'shop.admin.orders.confirm_order_payment_panel',
            function (Container $container): ConfirmOrderPaymentPanel {
                return new ConfirmOrderPaymentPanel(
                    $container->get('translator'),
                    $container->get('pongho_url'),
                    $container->get('nodetype_helper'),
                    $container->get('shop.domain.orders.commands.confirm_order_payment_factory'),
                    $container->get('logger'),
                );
            },
        );

        $container->share(
            'shop.domain.orders.commands.confirm_order_payment_factory',
            function (Container $container): ConfirmOrderPaymentCommandFactory {
                return new ConfirmOrderPaymentCommandFactory(
                    $container->get('translator'),
                );
            },
        );

        $container->share('shop.admin.orders.process_order_panel', function (Container $container): ProcessOrderPanel {
            return new ProcessOrderPanel(
                $container->get('translator'),
                $container->get('pongho_url'),
                $container->get('nodetype_helper'),
                $container->get('shop.domain.orders.commands.process_order_factory'),
                $container->get('logger'),
            );
        });

        $container->share(
            'shop.domain.orders.commands.process_order_factory',
            function (Container $container): ProcessOrderCommandFactory {
                return new ProcessOrderCommandFactory(
                    $container->get('shop_order_discounts'),
                    $container->get('shop.domain.orders.commands.process_order_notifier'),
                    $container->get('translator'),
                );
            },
        );

        $container->share(
            'shop.domain.orders.commands.process_order_notifier',
            function (Container $container): ProcessOrderNotifier {
                return new ProcessOrderNotifier(
                    $container->get(Mailer::class),
                    $container->get(Helper::class),
                    $container->get('site'),
                    $container->get('shop.domain.orders.commands.process_order_notifier.view'),
                    $container->get('language'),
                );
            },
        );

        $container->share(
            'shop.domain.orders.commands.process_order_notifier.view',
            function (Container $container): View {
                /** @var Site $site */
                $site = $container->get('site');

                $view = new View($site->getThemesPath('email/shop_order_processed.php'));
                $view->assignVars($container->get('global_view_vars'));

                return $view;
            },
        );
    }

    public function siteExtension(Site $site): void
    {
        $site->setDefaultOptions(Config::getDefault($site));
    }
}
