<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller;

use Application\Core\Controller;
use Application\Core\Mailer\Helper;
use Application\Core\Model\Address;
use Application\Core\Model\Country;
use Application\Core\Utilities\Validator;
use Application\Showcase\Exception\OrderException;
use Application\Showcase\Exception\ShippingAddressUndefinedException;
use Application\Showcase\Form\Builder\AddressFormBuilder;
use Application\Showcase\Model\Node;
use Application\Showcase\Model\Order;
use Application\Showcase\Model\Payment;
use Application\Showcase\Model\Shipping;
use Application\Showcase\Model\Size;
use Application\Showcase\Payment\ListenerResponse;
use Application\Showcase\Payment\PaymentOptions;
use Application\Showcase\Payment\TicketResponse;
use Pongho\Form\Form;
use Pongho\Http\Exception\HttpException;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\JsonResponse;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Response;
use Pongho\Template\View;
use Symfony\Component\Mailer\Mailer;
use Symfony\Component\Mime\Address as EmailAddress;
use Symfony\Component\Mime\Email;

/**
 * Controller per la gestione delle azioni dello shop.
 */
class ShopController extends Controller
{
    /**
     * Sito corrente.
     *
     * @var \Application\Core\Model\Site
     */
    protected $site;

    /**
     * Carrello.
     *
     * @var \Application\Showcase\Model\Order
     */
    protected $cart;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * {@inheritdoc}
     */
    public function handle()
    {
        $logger = $this->getLogger();
        $request = $this->getRequest();

        $logger->debug(sprintf('[REQUEST] %s', $request), ['ip' => $request->getClientIp()]);

        $response = parent::handle();

        $logger->debug(
            sprintf(
                '[RESPONSE] HTTP/%s %s %s',
                $response->getProtocolVersion(),
                $response->getStatusCode(),
                $response->getStatusMessage()
            )
        );

        return $response;
    }

    /**
     * Azione `cart`.
     *
     * Visualizza il carrello e risolve le richieste arrivate via POST.
     *
     * @return \Pongho\Http\Response
     */
    public function cartAction()
    {
        if (!$this->getHelper()->getUser()->hasPermit('shop.buy')) {
            throw new HttpNotFoundException();
        }

        $logger = $this->getLogger();
        $cart = $this->getCart();
        $request = $this->getRequest();
        $lang = $this->getHelper()->getLocalization();

        if ($request->getMethod() === 'POST') {
            $logger->info(
                '[CART] Updating cart.',
                [
                    'cart' => $cart->getId() ?: 'NULL',
                ]
            );

            // Elimino eventuali flag sulle righe del carrello
            /** @var \Application\Showcase\Model\OrderRow $row */
            foreach ($cart->getRows() as $row) {
                if ($row->is_unavailable) {
                    $logger->info(
                        '[CART] Row is unavailable, then delete it.',
                        [
                            'cart' => $cart->getId() ?: 'NULL',
                            'row'  => $row->id,
                        ]
                    );

                    $row->delete();
                } elseif ($row->is_previous_cart) {
                    $logger->info(
                        '[CART] Row was in previous cart, then set is_previous_cart to false.',
                        [
                            'cart' => $cart->getId() ?: 'NULL',
                            'row'  => $row->id,
                        ]
                    );

                    $row->is_previous_cart = false;
                    $row->save();
                }
            }

            // Quantità
            $quantities = (array) $request->post->get('quantities', []);
            $cart->changeQuantities($quantities);

            // Discounts
            $this->updateDiscounts();

            if ($cart->errors->is_empty()) {
                if ($request->isAjax()) {
                    $discounts = [];
                    foreach ($cart->getDiscounts() as $discount) {
                        $discounts[] = [
                            'name'        => $discount->name(),
                            'value'       => $discount->value(),
                            'formatValue' => $discount->formatValue(),
                        ];
                    }

                    $response = [
                        'cart' => [
                            'productsTotal' => $cart->productsTotal(),
                            'discounts'     => $discounts,
                            'subtotal'      => $cart->subtotal(),
                        ],
                    ];

                    return $this->getHelper()->displayJsonMessage($lang->get('cart_updated'), false, $response);
                }

                if ($cart->customerIsGuest()) {
                    return $this->getHelper()->redirectResponse('/shop/login/');
                }

                if ($request->post->has('coupon-submit') || $request->post->has('shopping-points-submit')) {
                    return $this->getHelper()->redirectResponse('/shop/cart/');
                }

                return $this->getHelper()->redirectResponse('/shop/checkout/');
            }
        }

        // Info
        $info = [];
        foreach ($cart->rows() as $row) {
            if ($row->is_previous_cart) {
                $info['cart_has_previous_rows'] = $lang->get('cart_has_previous_rows');
            }

            if ($row->is_unavailable || $row->availabilities() < $row->quantity()) {
                $row->is_unavailable = true;
                $row->quantity = 0;
                $row->save();
                $cart->save();

                $info['cart_has_unavailable_rows'] = $lang->get('cart_has_unavailable_rows');
            }
        }

        $this->getHelper()->getThemeHeaderHelper()
            ->setTitle($lang->get('page_title_cart'));

        $this->getHelper()->getTheme()
            ->setTemplate('shop/cart.php')
            ->assignVars([
                'action'              => 'cart',
                'order'               => $cart,
                'info'                => $info,
                'errors'              => $this->translateErrors($cart->errors),
                'taxation'            => $this->getContainer()->get('shop_taxation'),
                'checkout_breadcrumb' => $this->getContainer()->get('checkout_breadcrumb')->build($cart),
            ]);

        $this->displayJavascript();

        return null;
    }

    /**
     * Restituisce l’istanza del carrello per l’utente corrente.
     *
     * È possibile forzare il ri-caricamento del carrello se si presume che questo possa non
     * essere aggiornato. È comunque sconsigliato per motivi di performance.
     *
     * @param bool $force_update
     * @return \Application\Showcase\Model\Order
     */
    public function getCart($force_update = false)
    {
        if ($this->cart === null || $force_update) {
            $this->cart = $this->getContainer()->get('cart');
            $this->cart->setLogger($this->getLogger());
        }

        return $this->cart;
    }

    /**
     * Updates the discounts.
     */
    protected function updateDiscounts()
    {
        $request = $this->getRequest();
        $cart = $this->getCart();

        /**
         * @var \Application\Showcase\Discount\OrderDiscounts $orderDiscounts
         * @var \Application\Showcase\Discount\OrderDiscountsFacade $orderDiscountsFacade
         */
        $orderDiscounts = $this->getContainer()->get('shop_order_discounts');
        $orderDiscountsFacade = $this->getContainer()->get('shop_order_discounts_facade');

        // Coupons
        if (
            $orderDiscounts->isEnabled('coupon_code')
            && $request->post->has('coupon-code')
            && $this->getHelper()->getUser()->isLogged()
        ) {
            $couponCode = trim((string) $request->post->get('coupon-code'));
            $orderDiscountsFacade->applyCouponCode($cart, $couponCode);
        }

        // Shopping Points
        if (
            $orderDiscounts->isEnabled('shopping_points')
            && $request->post->has('shopping-points')
        ) {
            $points = (int) $request->post->get('shopping-points');
            $orderDiscountsFacade->updateShoppingPoints($cart, $points);
        }

        // Discount sort
        $orderDiscountsFacade->updateDiscounts($cart);

        // Earned Shopping Points
        $cart->updateTotal();
        $this->updateEarnedShoppingPoints();

        $cart->save();
    }

    /**
     * Carica il javascript per la gestione delle pagine dello shop.
     */
    protected function displayJavascript()
    {
        $order = $this->getCart();
        $site = $this->getHelper()->getSite();

        $options = [
            'subtotal' => $order->subtotal(),
        ];

        if ($site->getOption('enable_tax')) {
            $options['tax'] = [
                'enabled' => true,
                'value'   => $site->getOption('tax_value'),
            ];
        }

        $discounts = [];
        foreach ($order->getDiscounts() as $discount) {
            $discounts[] = [
                'name'        => $discount->name(),
                'value'       => $discount->value(),
                'formatValue' => $discount->formatValue(),
            ];
        }

        $options['cart'] = [
            'productsTotal' => $order->productsTotal(),
            'discounts'     => $discounts,
            'subtotal'      => $order->subtotal(),
        ];

        $this->getHelper()
            ->addJavascriptInline('var Pongho = Pongho || {}; Pongho.ShopOptions = ' . json_encode($options) . ';')
            ->addJavascript(
                pongho_url(
                    '/Application/Showcase/Resources/public/js/shop.min.js?v='
                    . filemtime(__DIR__ . '/../Resources/public/js/shop.min.js')
                )
            );
    }

    /**
     * Azione `addtocart`.
     *
     * Aggiunge un prodotto al carrello.
     *
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return \Pongho\Http\Response
     */
    public function addtocartAction()
    {
        if (!$this->getHelper()->getUser()->hasPermit('shop.buy')) {
            throw new HttpNotFoundException();
        }

        $request = $this->getRequest();

        // Quantità
        if ($request->query->has('quantity')) {
            $quantity = (int) $request->query->get('quantity');
        } elseif ($request->post->has('quantity')) {
            $quantity = (int) $request->post->get('quantity');
        } else {
            $quantity = 1;
        }

        if ($quantity <= 0) {
            $quantity = 1;
        }

        $productId = $request->post->has('product') ? $request->post->get('product') : (int) $this->getParameter('id');

        if (!$productId) {
            throw new HttpNotFoundException();
        }

        /** @var Node $product */
        $product = Node::find($productId);

        if (!$product) {
            throw new HttpNotFoundException();
        }

        $sizeId = null;
        if ($request->post->has('size')) {
            $sizeId = (int) $request->post->get('size');
        }

        /** @var \Application\Showcase\Model\Size $size */
        $size = $sizeId ? Size::find($sizeId) : null;

        // Aggiungo in carrello
        try {
            if (!$this->addProductToCart($product, $quantity, $size)) {
                return $this->getHelper()->displayError($this->getHelper()->getLocalization()->get('cart_row_not_added'));
            }
        } catch (OrderException $orderException) {
            return $this->getHelper()->displayError(
                $this->getHelper()->getLocalization()->get($orderException->getMessage())
            );
        }

        return new RedirectResponse('/shop/cart/');
    }

    /**
     * Permette di aggiungere un prodotto nel carrello.
     *
     * @param integer                          $quantity
     * @param \Application\Showcase\Model\Size $size
     * @return bool
     * @throws \Application\Showcase\Exception\OrderException Se ci sono problemi (generalmente di disponibilità).
     */
    protected function addProductToCart(Node $product, $quantity = 1, ?Size $size = null)
    {
        $logger = $this->getLogger();
        $cart = $this->getCart();

        $logger->info(
            sprintf('[%s] Adding product to the cart', strtoupper($this->action)),
            [
                'cart'     => $cart->getId() ?: 'NULL',
                'product'  => $product->id,
                'size'     => $size ? $size->id : null,
                'quantity' => $quantity,
            ]
        );

        $sizes = $product->getSizes(true);
        if ($sizes && !$size) {
            // Non posso inserire nel carrello un prodotto che prevede taglie senza sapere la taglia desiderata
            return false;
        }

        return $cart->transaction(
            function () use ($cart, $product, $quantity, $size) {
                if ($cart->addProduct($product, $quantity, $size) === false) {
                    return false;
                }

                $this->updateDiscounts();

                return true;
            }
        );
    }

    /**
     * Azione `removefromcart`.
     *
     * Rimuove un prodotto dal carrello.
     *
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return \Pongho\Http\Response
     */
    public function removefromcartAction()
    {
        $logger = $this->getLogger();
        $cart = $this->getCart();
        $row_id = $this->getParameter('id');

        $logger->info(
            '[REMOVE_FROM_CART] Removing the row from the cart.',
            [
                'cart' => $cart->getId() ?: 'NULL',
                'row'  => $row_id,
            ]
        );

        if (!is_numeric($row_id)) {
            $logger->warning(
                '[REMOVE_FROM_CART] The row ID is not numeric.',
                [
                    'cart' => $cart->getId() ?: 'NULL',
                    'row'  => $row_id,
                ]
            );

            throw new HttpNotFoundException();
        }

        if (!$cart->delRow($row_id)) {
            $logger->warning(
                '[REMOVE_FROM_CART] Could not delete the row.',
                [
                    'cart' => $cart->getId() ?: 'NULL',
                    'row'  => $row_id,
                ]
            );

            return $this->getHelper()->displayError($this->getHelper()->getLocalization()->get('cart_row_not_deleted'));
        }

        // Aggiorno gli sconti
        $this->updateDiscounts();

        $logger->debug(
            '[REMOVE_FROM_CART] The row has been deleted.',
            [
                'cart' => $cart->getId() ?: 'NULL',
                'row'  => $row_id,
            ]
        );

        if ($this->getRequest()->isAjax()) {
            return $this->getHelper()->displayJsonMessage($this->getHelper()->getLocalization()->get('cart_row_deleted'));
        }

        return new RedirectResponse('/shop/cart/');
    }

    /**
     * Azione `emptycart`.
     *
     * Svuota il carrello.
     *
     * @return \Pongho\Http\Response
     */
    public function emptycartAction()
    {
        $cart = $this->getCart();

        $cart->delRows();
        $cart->removeAllDiscounts();

        // Aggiorno i punti guadagnati
        $this->updateEarnedShoppingPoints();

        $cart->save();

        if ($this->getRequest()->isAjax()) {
            return $this->getHelper()->displayJsonMessage($this->getHelper()->getLocalization()->get('cart_emptied'));
        }

        return new RedirectResponse('/shop/cart/');
    }

    /**
     * Azione `login`.
     *
     * Esegue il login o l’iscrizione dell’utente in fase di checkout.
     *
     * @return \Pongho\Http\Response
     */
    public function loginAction()
    {
        if ($this->getHelper()->getUser()->isLogged()) {
            return $this->getHelper()->redirectResponse('/shop/cart/');
        }

        $logger = $this->getLogger();
        $cart = $this->getCart();

        // se il carrello è vuoto, torno al carrello
        if ($cart->isEmpty()) {
            // Può succedere quando dal carrello procedo al pagamento e devo fare login. Poi per qualche motivo
            // (magari da un'altra tab del browser) svuoto il carrello e riaggiorno la pagina di login. Siccome è un
            // caso quantomeno bizzaro, preferisco loggarlo come warning per valutarlo meglio.
            $logger->warning(
                '[LOGIN] The cart is empty, then redirect to cart.',
                ['cart' => $cart->getId() ?: 'NULL']
            );

            return $this->getHelper()->redirectResponse('/shop/cart/');
        }

        $form = $this->createLoginForm();
        $form->handleRequest($this->getRequest());

        if ($form->hasHandled() && !$form->hasErrors()) {
            /** @var \Application\Showcase\Form\Subject\LoginSubject $subject */
            $subject = $form->getSubject();

            // Aggrego i carrelli precedenti
            if ($this->mergeCarts($cart)) {
                $logger->notice(
                    '[LOGIN] Merge carts, then redirect to cart.',
                    ['cart' => $cart->getId() ?: 'NULL']
                );

                return $this->getHelper()->redirectResponse('/shop/cart/');
            }

            if ($subject->hasShippingAddressException()) {
                $logger->notice(
                    '[LOGIN] Shipping address problems.',
                    ['cart' => $cart->getId() ?: 'NULL']
                );

                return $this->getHelper()->redirectResponse('/shop/addresses/');
            }

            $logger->info(
                '[LOGIN] OK, redirect to checkout.',
                ['cart' => $cart->getId() ?: 'NULL']
            );

            return $this->getHelper()->redirectResponse('/shop/checkout/');
        }

        $this->getHelper()->getThemeHeaderHelper()
            ->setTitle($this->getHelper()->getLocalization()->get('page_title_login'));

        $this->getHelper()->getTheme()
            ->setTemplate('shop/login.php')
            ->assignVars([
                'action'              => 'login',
                'login_form'          => $form,
                'order'               => $this->getCart(),
                'taxation'            => $this->getContainer()->get('shop_taxation'),
                'checkout_breadcrumb' => $this->getContainer()->get('checkout_breadcrumb')->build($cart),
            ]);

        $this->displayJavascript();

        return null;
    }

    /**
     * @return \Pongho\Form\Form
     */
    protected function createLoginForm()
    {
        /** @var \Application\Showcase\Form\Builder\LoginFormBuilder $builder */
        $builder = $this->getContainer()->get('shop_login_form_builder');

        return $builder->build();
    }

    /**
     * Unisce il carrello corrente con quelli precedenti.
     *
     * @return bool Se eseguire un redirect al carrello o meno, per mostrare la modifica al cliente.
     */
    protected function mergeCarts(Order $cart)
    {
        $redirect_to_cart = false;

        foreach (Order::findPreviousCarts($this->getHelper()->getUser(), $this->getSession(), $this->getHelper()->getSite()) as $order) {
            /** @var \Application\Showcase\Model\OrderRow $row */
            foreach ($order->getRows(true) as $row) {
                if ($row->changeOrder($cart)) {
                    $row->is_previous_cart = true;

                    if ($row->save()) {
                        $redirect_to_cart = true;
                    }
                } else {
                    // Non ha senso spostare le quantità perché quasi sicuramente l’utente stava rifacendo lo stesso ordine.
                    $order->delete();
                }
            }

            $order->delete();
        }

        return $redirect_to_cart;
    }

    /**
     * Azione `addresses`.
     *
     * Visualizza la rubrica indirizzi per modificare l’indirizzo di spedizione.
     *
     * @return \Pongho\Http\Response
     * @throws \Pongho\Http\Exception\HttpUnauthorizedException
     */
    public function addressesAction()
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            throw new HttpUnauthorizedException();
        }

        $cart = $this->getCart();

        // se il carrello è vuoto, torno al carrello
        if ($cart->isEmpty()) {
            return $this->getHelper()->redirectResponse('/shop/cart/');
        }

        $form = $this->createAddressForm($this->createAddress(), '/shop/add_address/');

        /** @var \Application\Core\Model\Account $account */
        $account = $this->getHelper()->getUser()->getAccount();

        $this->getHelper()->getThemeHeaderHelper()
            ->setTitle($this->getHelper()->getLocalization()->get('page_title_addresses'));

        $this->getHelper()->getTheme()
            ->setTemplate('shop/addresses.php')
            ->assignVars([
                'action'              => 'addresses',
                'addresses'           => $account->addresses,
                'address_form'        => $form,
                'delete_address_url'  => false,
                'order'               => $this->getCart(),
                'taxation'            => $this->getContainer()->get('shop_taxation'),
                'checkout_breadcrumb' => $this->getContainer()->get('checkout_breadcrumb')->build($cart),
            ]);

        $this->displayJavascript();

        return null;
    }

    /**
     * Azione `edit_address`.
     *
     * Permette di gestire un indirizzo. Questa azione è pensata per funzionare solo in AJAX.
     *
     * @return \Pongho\Http\JsonResponse
     * @throws \Pongho\Http\Exception\HttpUnauthorizedException
     */
    public function edit_addressAction()
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            throw new HttpUnauthorizedException();
        }

        $address = $this->findAddress();
        $form = $this->createAddressForm($address, '/shop/edit_address/' . $address->id . '/');

        return $this->handleAddressForm($address, $form);
    }

    /**
     *  Azione `add_address`.
     *
     * Permette di aggiungere un indirizzo. Questa azione è pensata per funzionare solo in AJAX.
     *
     * @return JsonResponse|RedirectResponse
     * @throws \Pongho\Http\Exception\HttpUnauthorizedException
     */
    public function add_addressAction()
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            throw new HttpUnauthorizedException();
        }

        $address = $this->createAddress();
        $form = $this->createAddressForm($address, '/shop/add_address/');

        return $this->handleAddressForm($address, $form);
    }

    /**
     * Azione `useaddress`.
     *
     * Permette di impostare un indirizzo di spedizione.
     *
     * @return \Pongho\Http\Response
     * @throws \Pongho\Http\Exception\HttpException
     */
    public function use_addressAction()
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            throw new HttpUnauthorizedException();
        }

        $address = $this->findAddress();
        $this->assignAddressToCart($address);

        return $this->getHelper()->redirectResponse('/shop/checkout/');
    }

    /**
     * Azione `deleteaddress`.
     *
     * Permette di cancellare un indirizzo.
     *
     * @throws \RuntimeException
     * @throws \Pongho\Http\Exception\HttpException
     * @return \Pongho\Http\Response
     */
    public function delete_addressAction()
    {
        $address = $this->findAddress();
        $account = $this->getHelper()->getUser()->getAccount();

        if ($address->user_id !== $account->getId()) {
            throw new HttpUnauthorizedException();
        }

        if ($address->delete()) {
            return $this->getHelper()->redirectResponse('/shop/addresses/');
        }

        throw new \RuntimeException('Could not delete address!');
    }

    /**
     * @return Address
     */
    protected function createAddress()
    {
        $account = $this->getHelper()->getUser()->getAccount();

        /** @var \Application\Core\Model\LanguageSite $languageSite */
        $languageSite = $this->getContainer()->get('language_site');

        $countryCode = match ($languageSite->getLanguage()->getCulture()) {
            'en_US' => 'US',
            default => 'IT',
        };

        /**
         * @var Country $country
         */
        $country = Country::first([
            'conditions' => ['code = :code', 'code' => $countryCode],
        ]);

        return new Address([
            'user_id'    => $account->getId(),
            'country_id' => $country->id,
        ]);
    }

    /**
     * @return Address
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    protected function findAddress()
    {
        /** @var \Application\Core\Model\Address $address */
        $address = Address::find($this->getParameter('id'));

        if ($address === null) {
            throw new HttpNotFoundException();
        }

        return $address;
    }

    /**
     * @param string  $action
     * @return \Pongho\Form\Form
     */
    protected function createAddressForm(Address $address, $action)
    {
        $builder = new AddressFormBuilder($address, $this->getHelper()->getLocalization());

        $form = $builder->build();
        $form->setAction(url($action));
        $form->addCssClass('shop-address-form');

        return $form;
    }

    /**
     * @return JsonResponse|RedirectResponse
     */
    protected function handleAddressForm(Address $address, Form $form)
    {
        $form->handleRequest($this->getRequest());

        if ($form->hasHandled() && !$form->hasErrors()) {
            // Associo l’indirizzo appena creato / modificato al carrello.
            $this->assignAddressToCart($address);

            return $this->getHelper()->redirectResponse('/shop/checkout/');
        }

        $delete_url = false;
        if ($this->getAction() === 'edit_address') {
            $delete_url = url('/shop/delete_address/' . $address->id . '/');
        }

        $this->getHelper()->getTheme()
            ->setTemplate('shop/address-form.php')
            ->assignVars([
                'address'            => $address,
                'address_form'       => $form,
                'delete_address_url' => $delete_url,
            ]);

        $response = [
            'html' => $this->getHelper()->getTheme()->render(),
        ];

        return new JsonResponse($response);
    }

    protected function assignAddressToCart(Address $address)
    {
        /** @var \Application\Core\Model\Account $account */
        $account = $this->getHelper()->getUser()->getAccount();
        $cart = $this->getCart();

        if ($account->shipping_address_id === null) {
            $account->shipping_address_id = $address->id;
            $account->save(false);
        }

        $cart->shipping_address_id = $address->id;
        $cart->shipping_address = $address;
        $cart->populateShippingAddressAttributes();
        $cart->populateInvoiceInfoAttributes();

        $cart->save(false);
    }

    /**
     * Azione `checkout`.
     *
     * Permette di selezionare il tipo di pagamento e di spedizione.
     * All’invio del modulo istanzia l’handler del tipo di pagamento e lascia fare a lui.
     *
     * @return Response
     * @throws \RuntimeException Se non sono definiti i metodi di pagamento o di spedizione.
     * @throws \Exception Può essere lanciata dall'handler del metodo di pagamento.
     *
     * @todo Internazionalizzare la gestione del codice cliente.
     */
    public function checkoutAction()
    {
        $logger = $this->getLogger();
        $cart = $this->getCart();

        // Se il carrello è vuoto, torno al carrello
        if ($cart->isEmpty()) {
            $logger->warning(
                '[CHECKOUT] The cart is empty, then redirect to cart.',
                ['cart' => $cart->getId() ?: 'NULL']
            );

            return new RedirectResponse('/shop/cart/');
        }

        // Se l’utente non è loggato vado alla pagina di login
        if (!$cart->customerIsGuest() && !$this->getHelper()->getUser()->isLogged()) {
            $logger->warning(
                '[CHECKOUT] The user is not logged, then redirect to login.',
                ['cart' => $cart->getId() ?: 'NULL']
            );

            return new RedirectResponse('/shop/login/');
        }

        /** @var \Application\Showcase\Utilities\CheckoutOptions $checkout_options */
        $checkout_options = $this->getContainer()->get('shop_checkout_options');

        $this->getHelper()->notify($this, 'shop.checkout.start', ['cart' => $cart]);

        // Indirizzo
        if ($checkout_options->isStandardShippingMethodsEnabled() && !$cart->customerIsGuest()) {
            try {
                $cart->populateShippingAddressAttributes();
                $cart->populateInvoiceInfoAttributes();
            } catch (ShippingAddressUndefinedException $e) {
                $logger->notice(
                    '[CHECKOUT] Shipping address problems: ' . $e->getMessage(),
                    ['cart' => $cart->getId() ?: 'NULL']
                );

                return $this->getHelper()->redirectResponse('/shop/addresses/');
            }
        }

        /** @var Payment $payment */
        $payments = [];
        foreach (Payment::allEnabled() as $payment) {
            $payments[$payment->id] = $payment;
        }

        if ($payments === []) {
            $logger->critical(
                '[CHECKOUT] No payments defined!',
                ['cart' => $cart->getId() ?: 'NULL']
            );

            throw new \RuntimeException('No payments defined!');
        }

        $payments = $this->getHelper()->filter($this, 'shop.checkout.filter_payments', $payments, [
            'cart' => $cart,
        ]);

        /** @var Shipping $shipping */
        $shippings = [];
        foreach (Shipping::all() as $shipping) {
            $shippings[$shipping->id] = $shipping;
        }

        if ($shippings === []) {
            $logger->critical(
                '[CHECKOUT] No shippings defined!',
                ['cart' => $cart->getId() ?: 'NULL']
            );

            throw new \RuntimeException('No shippings defined!');
        }

        $shippings = $this->getHelper()->filter($this, 'shop.checkout.filter_shippings', $shippings, [
            'cart' => $cart,
        ]);

        $request = $this->getRequest();
        $continue = true;

        $this->getHelper()->notify($this, 'shop.checkout.before_handle_form', ['cart' => $cart]);

        if ($request->getMethod() === 'POST') {
            $post_order = $request->post->get('order');

            if (!isset($post_order['conditions']) || !$post_order['conditions']) {
                $cart->addError('conditions', 'Devi accettare le condizioni per proseguire.');
            }

            if (isset($post_order['payment_id']) && isset($payments[$post_order['payment_id']])) {
                $cart->setPayment($payments[$post_order['payment_id']]);
            } else {
                $cart->setPayment(reset($payments));
            }

            if (isset($post_order['shipping_id']) && isset($shippings[$post_order['shipping_id']])) {
                $cart->setShipping($shippings[$post_order['shipping_id']]);
            } else {
                $cart->setShipping(reset($shippings));
            }

            if ($this->getHelper()->getSite()->getOption('enable_order_note')) {
                $cart->note = isset($post_order['note']) ? trim($post_order['note']) : '';
            }

            // Discounts
            $this->updateDiscounts();

            // Se l'utente ha cliccato su "Ricalcola", deve rimanere in questa pagina
            if ($request->post->has('coupon-submit') || $request->post->has('shopping-points-submit')) {
                $continue = false;
            }

            $this->getHelper()->notify($this, 'shop.checkout.submit', ['cart' => $cart]);

            if (!$cart->errors->is_empty()) {
                $this->getHelper()->getTheme()
                    ->assignVar('errors', $this->translateErrors($cart->errors));
            } elseif ($continue && $this->resolveCustomerCode($cart)) {
                $cart->updateTotal();

                $handler = $this->getPaymentHandler($cart);
                $handler->preparePayment();

                $logger->debug(
                    '[CHECKOUT] Order submit.',
                    [
                        'order' => $cart->attributes(),
                    ]
                );

                if ($cart->save()) {
                    // Sposto i prodotti legati al carrello nei legati al pagamento.
                    $cart->updateBounds();

                    try {
                        return $handler->handlePayment(new PaymentOptions('/shop/'))->getHttpResponse();
                    } catch (\Exception $e) {
                        $logger->critical('[CHECKOUT] ' . $e->getMessage(), ['exception' => $e]);

                        throw $e;
                    }
                }
            }
        } else {
            if (!$cart->payment_id) {
                $cart->payment_id = reset($payments)->id;
            }

            if (!$cart->shipping_id) {
                $cart->shipping_id = reset($shippings)->id;
            }
        }

        $this->getHelper()->getThemeHeaderHelper()
            ->setTitle($this->getHelper()->getLocalization()->get('page_title_checkout'));

        $this->getHelper()->getTheme()
            ->setTemplate('shop/checkout.php')
            ->assignVars([
                'action'              => 'checkout',
                'order'               => $cart,
                'payments'            => $payments,
                'shippings'           => $shippings,
                'taxation'            => $this->getContainer()->get('shop_taxation'),
                'checkout_breadcrumb' => $this->getContainer()->get('checkout_breadcrumb')->build($cart),
                'checkout_options'    => $checkout_options,
            ]);

        $this->displayJavascript();

        $this->getHelper()->notify($this, 'shop.checkout.render', ['cart' => $cart]);

        return null;
    }

    /**
     * Aggiorna il conteggio dei punti guadagnati.
     */
    protected function updateEarnedShoppingPoints()
    {
        /** @var \Application\Showcase\Discount\OrderDiscounts $orderDiscounts */
        $orderDiscounts = $this->getContainer()->get('shop_order_discounts');

        if (!$orderDiscounts->isEnabled('shopping_points')) {
            return;
        }

        /** @var \Application\Showcase\ShoppingPoints\Scheme $scheme */
        $scheme = $this->getContainer()->get('shop_shopping_points_scheme');

        $this->getCart()->updateEarnedShoppingPoints($scheme);
    }

    /**
     * Restituisce il gestore del pagamento per l’ordine passato.
     *
     * @return \Application\Showcase\Payment\BasePayment
     */
    protected function getPaymentHandler(Order $order)
    {
        $handler = $order->getPaymentHandler()
            ->setContainer($this->getContainer())
            ->setDebug($this->getHelper()->isDebug());

        return $this->getHelper()->filter(
            $this,
            'shop.filter_payment_handler',
            $handler,
            ['order' => $order]
        );
    }

    /**
     * Risolve il campo con il codice cliente (Partita IVA, Codice Fiscale, o equivalente).
     *
     * @return bool
     *
     * @todo I18n
     * @todo Utilizzare il servizio request al posto della variabile $_POST
     */
    protected function resolveCustomerCode(Order $cart)
    {
        $is_valid = true;

        if (isset($_POST['order']['customer_code_name']) && isset($_POST['order']['customer_code_value'])) {
            $customer_code_name = trim((string) $_POST['order']['customer_code_name']);
            $customer_code_value = trim((string) $_POST['order']['customer_code_value']);

            // Se c'è il campo, significa che è obbligatorio, quando un
            // cliente ci chiederà diversamente ci muoveremo di conseguenza.
            if ($customer_code_value === '') {
                $is_valid = false;

                $cart->addError('customer_code_value', 'required_customer_code');
            }

            // @todo Internazionalizzazione dei nomi.
            if (!in_array($customer_code_name, ['partita_iva', 'codice_fiscale'])) {
                $is_valid = false;

                $cart->addError('customer_code_value', 'invalid_customer_code');
            }

            if (
                $is_valid
                && method_exists(\Application\Core\Utilities\Validator::class, $customer_code_name)
                && !Validator::$customer_code_name($customer_code_value)
            ) {
                $is_valid = false;
                $cart->addError('customer_code_value', 'invalid_customer_code');
            }

            if ($is_valid) {
                $cart->customer_code_name = $customer_code_name;
                $cart->customer_code_value = $customer_code_value;

                if (!$cart->customerIsGuest() && $cart->customer->attributePresent($customer_code_name) && !$cart->customer->$customer_code_name) {
                    $cart->customer->$customer_code_name = $customer_code_value;
                    $cart->customer->save();
                }
            }
        }

        return $is_valid;
    }

    /**
     * Azione `ticket`.
     *
     * Visualizza uno scontrino dell’ordine appena fatto.
     *
     * @throws \LogicException
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return \Pongho\Http\Response
     */
    public function ticketAction()
    {
        $request = $this->getRequest();
        $logger = $this->getLogger();

        /** @var \Application\Showcase\Model\Order $order */
        if (($order = Order::find($this->getParameter('id', null, true))) === null) {
            // Mi viene in mente solo qualcuno che vuole fare il furbo.
            $logger->warning('[TICKET] Order not found.');

            throw new HttpNotFoundException();
        }

        /** @var \Application\Showcase\Utilities\CheckoutOptions $checkout_options */
        $checkout_options = $this->getContainer()->get('shop_checkout_options');
        $order->setCheckoutOptions($checkout_options);

        // Se l’utente non è loggato vado alla pagina di login
        if (!$order->customerIsGuest() && !$this->getHelper()->getUser()->isLogged()) {
            $logger->warning(
                '[TICKET] The user is not logged, then redirect to login.',
                ['order' => $order->getId()]
            );

            return new RedirectResponse('/shop/login/');
        }

        $hash = $request->query->get('hash');
        if (!$hash || $order->payment_hash !== $hash) {
            $logger->warning(
                '[TICKET] Order hash not found, or mismatch.',
                [
                    'order'        => $order->getId(),
                    'order_hash'   => $order->payment_hash,
                    'request_hash' => $hash,
                ]
            );

            throw new HttpNotFoundException();
        }

        // Ottengo la risposta dall’handler e la controllo.
        $handler = $this->getPaymentHandler($order);
        $response = $handler->handleTicket();

        if (!$response instanceof TicketResponse) {
            $message = sprintf('Payment handler "%s" not return a valid ticket response!', $handler::class);
            $logger->critical('[TICKET] ' . $message, ['order' => $order->getId()]);

            throw new \LogicException($message);
        }

        // Posso confermare l’ordine?
        if ($response->isOrderConfirmed()) {
            $logger->debug('[TICKET] Order confirmed.', ['order' => $order->getId()]);

            $this->confirmOrder($order);
        }

        $order->save();

        $http_response = $response->getResponse();

        if ($http_response instanceof Response) {
            return $http_response;
        }

        $this->getHelper()->getThemeHeaderHelper()
            ->setTitle($this->getHelper()->getLocalization()->get('page_title_ticket'));

        // Vista
        $this->getHelper()->getTheme()
            ->setTemplate('shop/ticket.php')
            ->assignVars([
                'action'              => 'ticket',
                'order'               => $order,
                'taxation'            => $this->getContainer()->get('shop_taxation'),
                'checkout_breadcrumb' => $this->getContainer()->get('checkout_breadcrumb')->build($order),
                'checkout_options'    => $checkout_options,
            ]);

        return null;
    }

    /**
     * Azione `listener`.
     *
     * Alcuni metodi di pagamento richiedono un indirizzo che gestisca le notifiche sulle modifiche fatte agli ordini.
     * Questa azione si occupa di "ascoltare" queste notifiche interfacciandosi con l’handler del metodo di pagamento.
     *
     * @throws \Exception
     * @throws \Pongho\Http\Exception\HttpException
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return \Pongho\Http\Response
     */
    public function listenerAction()
    {
        $logger = $this->getLogger();

        /** @var Payment $payment */
        if (($payment = Payment::find($this->getParameter('id', null, true))) === null) {
            $logger->warning('[LISTENER] Payment not found.');

            throw new HttpNotFoundException();
        }

        $handler_class = $payment->handler_class;

        /** @var \Application\Showcase\Payment\BasePayment $handler */
        $handler = new $handler_class(null, $this->getContainer(), $this->getHelper()->isDebug());

        try {
            // Ottengo la risposta dall’handler e la controllo.
            $response = $handler->handleListener($this->getRequest(), new PaymentOptions('/shop/'));

            if (!$response instanceof ListenerResponse) {
                $message = sprintf('Payment handler "%s" not return a valid listener response!', $handler::class);
                $logger->warning('[LISTENER] ' . $message);

                throw new \LogicException($message);
            }

            $order = $handler->getOrder();
        } catch (HttpException $e) {
            // Problemi HTTP.
            $order = $handler->getOrder(false);
            $order_id = $order ? $order->getId() : 'NULL';

            $logger->warning('[LISTENER] HTTP Exception.', ['order' => $order_id]);

            throw $e;
        } catch (\BadMethodCallException) {
            // Listener non supportato.
            $order = $handler->getOrder(false);
            $order_id = $order ? $order->getId() : 'NULL';

            $logger->warning(
                sprintf('[LISTENER] Handler "%s" does not support the listener.', $handler_class),
                [
                    'order' => $order_id,
                ]
            );

            throw new HttpNotFoundException();
        } catch (\Exception $e) {
            // Qualcosa è andato storto...
            $order = $handler->getOrder(false);
            $order_id = $order ? $order->getId() : 'NULL';

            $logger->error(
                '[LISTENER] Exception: ' . $e->getMessage(),
                [
                    'order'     => $order_id,
                    'exception' => $e,
                ]
            );

            throw new HttpException(400);
        }

        // Devo confermare l’ordine?
        if ($response->isOrderConfirmed()) {
            $logger->debug(
                '[LISTENER] Order confirmed.',
                [
                    'order'   => $order->getId(),
                    'handler' => $handler_class,
                ]
            );

            $this->confirmOrder($order);
        } elseif ($response->haveToSendFailedNotification()) {
            $logger->debug(
                '[LISTENER] Payment failed.',
                [
                    'order'   => $order->getId(),
                    'handler' => $handler_class,
                ]
            );

            $this->sendOrderFailedNotification($order);
        }

        $order->save();

        return $response->getResponse();
    }

    /**
     * Conferma l’ordine.
     */
    protected function confirmOrder(Order $order)
    {
        if ($order->notify_sent === false) {
            $this->sendOrderEmails($order);
        }

        $order->notify_sent = true;

        // Aggiorna le disponibilità
        $order->save();
        $order->updateBounds();

        $this->getHelper()->notify($this, 'shop.order_confirmed', ['order' => $order]);
    }

    /**
     * Invia le notifiche dell’ordine eseguito.
     *
     * @throws \Exception
     */
    protected function sendOrderEmails(Order $order)
    {
        /** @var \Application\Showcase\Utilities\OrderNotification $notification */
        $notification = $this->getContainer()->get('shop_order_notification');

        $notification->send($order);
    }

    /**
     * Azione `paymentcancel`.
     *
     * Annulla il pagamento in corso.
     *
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return \Pongho\Http\Response
     */
    public function paymentcancelAction()
    {
        $logger = $this->getLogger();

        /** @var Order $order */
        $order = Order::find($this->getParameter('id', null, true));

        if (null === $order) {
            $logger->warning('[PAYMENT CANCEL] Order not found.');

            throw new HttpNotFoundException();
        }

        if ($order->isPaid()) {
            $logger->warning('[PAYMENT CANCEL] Order paid.');

            throw new HttpNotFoundException();
        }

        $response = $this->getPaymentHandler($order)->handleCancel('/shop/cart/');

        if ($order->paid_at === null || $order->paid_at->isNull()) {
            $order->save();
        }

        if ($response->getSendNotifyToAdmin()) {
            $this->sendOrderFailedNotification($order);
        }

        $order->updateBounds();

        return $response->getHttpResponse();
    }

    /**
     * Azione `orders`.
     *
     * Visualizza l’elenco degli ordini fatti dall’utente.
     *
     * @return \Pongho\Http\Response
     */
    public function ordersAction()
    {
        $user = $this->getHelper()->getUser();

        if (!$user->isLogged()) {
            $this->getSession()
                ->set('redirect', '/shop/orders/');

            return new Redirectresponse('/user/login/');
        }

        if (!$user->hasPermit('shop.view_orders')) {
            throw new HttpNotFoundException();
        }

        $orders = Order::all([
            'conditions' => [
                "site_id = :site AND customer_id = :customer AND status IN ('" . implode("', '", Order::$USER_ORDERS_STATUSES) . "')",
                'site'     => $this->getHelper()->getSite()->getId(),
                'customer' => $this->getHelper()->getUser()->getAccount()->getId(),
            ],
            'order'      => 'ordered_at DESC',
        ]);

        $this->getHelper()->getThemeHeaderHelper()
            ->setTitle($this->getHelper()->getLocalization()->get('page_title_orders'));

        // vista
        $this->getHelper()->getTheme()
            ->setTemplate('shop/orders.php')
            ->assignVars([
                'action'           => 'orders',
                'orders'           => $orders,
                'taxation'         => $this->getContainer()->get('shop_taxation'),
                'checkout_options' => $this->getContainer()->get('shop_checkout_options'),
            ]);

        return null;
    }

    /**
     * Azione `order`.
     *
     * Visualizza i dettagli di un ordine dell'utente.
     *
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return \Pongho\Http\Response
     */
    public function orderAction()
    {
        $user = $this->getHelper()->getUser();
        $id = $this->getParameter('id', null, true);

        if (!$user->isLogged()) {
            $this->getSession()
                ->set('redirect', "/shop/order/{$id}/");

            return new Redirectresponse('/user/login/');
        }

        if (!$user->hasPermit('shop.view_orders')) {
            throw new HttpNotFoundException();
        }

        /** @var \Application\Showcase\Model\Order $order */
        $order = Order::find($id);

        if ($order === null || $order->customer_id != $this->getHelper()->getUser()->getAccount()->getId()) {
            throw new HttpNotFoundException();
        }

        $this->getHelper()->getThemeHeaderHelper()
            ->setTitle(
                sprintf(
                    $this->getHelper()->getLocalization()->get('page_title_order'),
                    $order->id,
                    $order->created_at->format('d/m/Y')
                )
            );

        // vista
        $this->getHelper()->getTheme()
            ->setTemplate('shop/order.php')
            ->assignVars([
                'action'           => 'order',
                'order'            => $order,
                'taxation'         => $this->getContainer()->get('shop_taxation'),
                'checkout_options' => $this->getContainer()->get('shop_checkout_options'),
            ]);

        return null;
    }

    /**
     * @return \Psr\Log\LoggerInterface
     */
    private function getLogger()
    {
        if (!$this->logger) {
            $this->logger = $this->getContainer()->get('logger');
        }

        return $this->logger;
    }

    private function sendOrderFailedNotification(Order $order)
    {
        if ($order->isPaymentFailNotified()) {
            return;
        }

        try {
            $site = $this->getHelper()->getSite();

            $mailView = new View($site->getThemesPath('email/shop_payment_fail.php'));
            $mailView
                ->assignVars($this->getContainer()->get('global_view_vars'))
                ->assignVars([
                    'order' => $order,
                ]);

            $ordersEmail = $site->getOption('orders_email', $site->getOption('company_email'));

            /**
             * @var Mailer $mailer
             * @var Helper $helper
             */
            $mailer = $this->getContainer()->get(Mailer::class);
            $helper = $this->getContainer()->get(Helper::class);

            $email = (new Email())
                ->from(new EmailAddress($ordersEmail, $site->getOption('company_name')))
                ->to(new EmailAddress($ordersEmail, $site->getOption('company_name')))
                ->replyTo(new EmailAddress($order->getCustomerEmail(), $order->getCustomerName()))
                ->subject(sprintf(
                    $this->getHelper()->getLocalization()->get('email_shop_payment_fail_subject'),
                    $site->getName()
                ))
                ->html(
                    $helper->replaceVars(
                        trim(file_get_contents($site->getThemesPath('email/base.php'))),
                        [
                            'CONTENT' => $mailView->render(),
                        ]
                    )
                );

            $mailer->send($helper->prepareEmail($email));

            $this->getLogger()->debug(
                '[PAYMENT CANCEL] Sent email to admin.',
                [
                    'order' => $order->getId(),
                    'from'  => $ordersEmail,
                    'to'    => $ordersEmail,
                ]
            );

            $order->setPaymentFailNotified();
            $order->save();
        } catch (\Throwable $throwable) {
            $this->getLogger()->error(
                '[PAYMENT CANCEL] ' . $throwable->getMessage(),
                [
                    'order'     => $order->getId(),
                    'exception' => $throwable,
                ]
            );
        }
    }
}
