<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller;

use Application\Cms\Controller\NodeController as Controller;
use Application\Cms\Model\Term;
use Application\Showcase\Model\Helper\ProductUrlGenerator;
use Application\Showcase\Model\Node;
use Application\Showcase\Model\OrderDiscount;
use Pongho\Http\Response;
use Pongho\Utilities\DateTime;

/**
 * Controller per la gestione dei prodotti in frontend.
 */
class NodeController extends Controller
{
    /**
     * Modello del nodo.
     *
     * @var Node
     */
    private $node;

    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        parent::boot();

        $this->getHelper()->getTheme()
            ->assignVars([
                'taxation' => $this->getContainer()->get('shop_taxation'),
            ]);
    }

    /**
     * {@inheritdoc}
     */
    public function nodeAction()
    {
        $response = parent::nodeAction();

        if ($response instanceof Response) {
            return $response;
        }

        $this->getHelper()
            ->addJavascript(
                pongho_url(
                    '/Application/Showcase/Resources/public/js/shop.min.js?v='
                    . filemtime(__DIR__ . '/../Resources/public/js/shop.min.js')
                )
            );

        return null;
    }

    /**
     * {@inheritdoc}
     */
    protected function displayArchive($page = 1, ?Term $term = null)
    {
        $this->getHelper()
            ->addJavascript(
                pongho_url(
                    '/Application/Showcase/Resources/public/js/shop.min.js?v='
                    . filemtime(__DIR__ . '/../Resources/public/js/shop.min.js')
                )
            );

        return parent::displayArchive($page, $term);
    }

    /**
     * {@inheritdoc}
     *
     * @todo  Disponibilità sul nodo
     * @fixme La query non funziona: se metto una taglia in carrello, scompare l'intero prodotto, nonostante siano presenti disponibilità per altre taglie.
     */
    protected function getArchiveQueryOptions($page = 1, ?Term $term = null)
    {
        $options = parent::getArchiveQueryOptions($page, $term);
        $site_module = $this->getSiteModule();

        if ($site_module->getOption('enable_availabilities') && $site_module->getOption('hide_unavailable_from_archives')) {
            if ($site_module->getOption('enable_sizes')) {
                $conditions = [
                    "CASE WHEN `from`.bound_in_cart_updated_at IS NOT NULL OR `from`.bound_in_cart_updated_at > :cart_expiry THEN `from`.bound_in_cart ELSE 0 END",
                    "CASE WHEN `from`.bound_in_payment_updated_at IS NOT NULL OR `from`.bound_in_payment_updated_at > :payment_expiry THEN `from`.bound_in_payment ELSE 0 END",
                    "`from`.bound_in_order",
                ];

                $options = Node::addCondition(
                    $options,
                    [
                        '`from`.availabilities > (' . implode(' + ', $conditions) . ')',
                        'cart_expiry'    => new DateTime(Node::BOUND_IN_CART_EXPIRY),
                        'payment_expiry' => new DateTime(Node::BOUND_IN_PAYMENT_EXPIRY),
                    ]
                );
            } else {
                // @todo Disponibilità sul nodo
            }
        }

        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClassName()
    {
        return Node::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getNode()
    {
        if ($this->node === null && ($node = parent::getNode()) !== null) {
            $this->node = new Node([
                'module_site_id' => $node->module_site_id,
                'language_id'    => $node->language_id,
            ]);

            $this->node->cast($node);

            $coupon = $this->getCoupon();
            if ($coupon) {
                $urlGenerator = new ProductUrlGenerator();
                $urlGenerator->setCoupon($coupon);

                $this->node->setUrlGenerator($urlGenerator);
            }
        }

        return $this->node;
    }

    /**
     * {@inheritdoc}
     */
    protected function filterNodes(array $nodes, $node_type, ?Term $term = null, $page = 1)
    {
        $coupon = $this->getCoupon();

        if ($coupon) {
            /** @var Node $node */
            foreach ($nodes as $node) {
                $urlGenerator = new ProductUrlGenerator();
                $urlGenerator->setCoupon($coupon);

                $node->setUrlGenerator($urlGenerator);
            }
        }

        return parent::filterNodes($nodes, $node_type, $term, $page);
    }

    /**
     * @return OrderDiscount|null
     */
    protected function getCoupon()
    {
        $request = $this->getRequest();
        $couponCode = trim((string) $request->query->get('coupon'));

        if (!$couponCode) {
            return null;
        }

        return OrderDiscount::findByCode($couponCode);
    }
}
