<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\CrudController;
use Application\Admin\Form\SelectHierarchyField;
use Application\Cms\Model\Node;
use Application\Showcase\Model\NodeType;
use Application\Showcase\Model\SizeName;
use Application\Showcase\Model\SizeType;
use Pongho\Http\Exception\HttpUnauthorizedException;

/**
 * Controller per la gestione delle taglie.
 */
class SizesController extends CrudController
{
    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        parent::boot();

        // Potrei aver impostato la risposta con il redirect alla pagina di login.
        if ($this->response) {
            return;
        }

        // Permesso consentito?
        // TODO: FS#454 - Permessi taglie
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.sizes')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    public function displayArchive($page = 1)
    {
        $path = $this->getParameter('path');

        /** @var \Application\Showcase\Model\NodeType $node_type */
        $node_type = $this->getHelper()->getNodeType();
        $rows = $node_type->getSizeTypes();

        foreach ($rows as $row) {
            $actions = [
                ['edit', 'href' => $this->url("/{$path}/edit/{$row->id}/"), 'class' => 'edit'],
            ];

            if ($row->isDeletable()) {
                $actions[] = ['delete', 'href' => $this->url("/{$path}/delete/{$row->id}/"), 'class' => 'delete'];
            }

            $row->actions = $this->parseActions($actions);
        }

        $this->getHelper()
            ->addCss(pongho_url(
                '/Application/Admin/Resources/views/css/style-legacy.css?v='
                . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/css/style-legacy.css')
            ));

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/sizes_list.php')
            ->assignVars([
                'rows'    => $rows,
                'add_url' => $this->url("/{$path}/add/"),
            ]);
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return SizeType::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getTitle($model)
    {
        $translator = $this->getHelper()->getTranslator();

        if ($this->getAction() === 'add') {
            return $translator->trans('Add size');
        }

        return $translator->trans('Edit the size “%title%”', ['%title%' => $model->title]);
    }

    /**
     * {@inheritdoc}
     */
    public function getFields($model)
    {
        $size_guide_node_type = $this->getHelper()->getNodeType()->getOption('size_guide_node_type');

        $options = [
            'conditions' => [
                "`from`.module_site_id = :module_site AND `from`.status IN ('publish', 'home') AND t.is_enabled = :enabled",
                'module_site' => $size_guide_node_type,
                'enabled'     => true,
            ],
        ];

        $size_guide_nodes = [];
        /** @var Node $node */
        foreach (Node::all($options, $this->getHelper()->getLanguage()->language_id) as $node) {
            $size_guide_nodes[$node->parent_id][$node->id] = $node->title;
        }

        $fields = [
            ['title'],
            [
                'node_id',
                'class' => SelectHierarchyField::class,
                'items' => $size_guide_nodes,
            ],
        ];

        return $this->getHelper()->filter($this, 'admin.showcase.sizes.filter_fields', $fields);
    }

    /**
     * Callback 'before_set_template'.
     *
     * @param \Application\Showcase\Model\SizeType $model
     */
    public function beforeSetTemplateCallback($model)
    {
        $sizes = '';
        foreach ($model->getSizes() as $size) {
            $sizes .= $this->generateRow($size);
        }

        $size_guide_node_type = $this->getHelper()->getNodeType()->getOption('size_guide_node_type');

        $this->getHelper()->getBodyView()
            ->assignVars([
                'sizes'              => '<ol id="sizes-sortable" class="sizes-sortable">' . $sizes . '<li class="blank-item">' . $this->getRowTemplate() . '</li></ol>',
                'size_guide_enabled' => !empty($size_guide_node_type),
            ]);

        $this->getHelper()
            ->addCss(pongho_url(
                '/Application/Admin/Resources/views/css/style-legacy.css?v='
                . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/css/style-legacy.css')
            ))
            ->addCss(pongho_url(
                '/Application/Showcase/Resources/views/css/sizes.css?v='
                . filemtime(PONGHO_PATH . '/Application/Showcase/Resources/views/css/sizes.css')
            ))
            ->addJavascript(pongho_url(
                '/Application/Showcase/Resources/views/js/jquery.sizes.js?v='
                . filemtime(PONGHO_PATH . '/Application/Showcase/Resources/views/js/jquery.sizes.js')
            ));
    }

    /**
     * Callback 'before_save'.
     *
     * @param \Application\Showcase\Model\SizeType $model
     */
    public function beforeSaveCallback($model)
    {
        $model->module_site_id = $this->getSiteModuleId();
    }

    /**
     * Callback 'after_save'.
     *
     * @param \Application\Showcase\Model\SizeType $model
     */
    public function afterSaveCallback($model)
    {
        $post_sizes = isset($_POST['sizes']) && is_array($_POST['sizes']) ? $_POST['sizes'] : [];

        foreach ($post_sizes as $size_id => &$attributes) {
            if (empty($attributes['name'])) {
                unset($post_sizes[$size_id]);

                continue;
            }

            $attributes['position'] = isset($attributes['position']) ? intval($attributes['position']) : 0;
            $attributes['name'] = isset($attributes['name']) ? html_escape(trim((string) $attributes['name'])) : '';

            if (is_numeric($size_id)) {
                $attributes['id'] = $size_id;
            }
        }

        $model->setSizes($post_sizes);
    }

    /**
     * Genera una riga.
     *
     * @return string
     */
    private function generateRow(SizeName $size)
    {
        return '<li id="size-item-' . $size->id . '" data-id="' . $size->id . '">' . $this->getRowTemplate($size) . '</li>';
    }

    /**
     * Restituisce il template per la singola riga.
     *
     * @param \Application\Showcase\Model\SizeName $size
     * @return string
     */
    private function getRowTemplate(?SizeName $size = null)
    {
        $translator = $this->getHelper()->getTranslator();

        if ($size instanceof SizeName) {
            $size_id = $size->id;
            $size_name = $size->name;
        } else {
            $size_id = 'new';
            $size_name = '';
        }

        return <<<EOT

<div class="size-item">
	<div class="size-handle"></div>
	<div class="size-item-fields">
		<input type="text" name="sizes[{$size_id}][name]" value="{$size_name}" placeholder="{$translator->trans('Name')}" class="input_text">
		<a class="size-item-delete">{$translator->trans('Delete')}</a>
	</div>
</div>

EOT;
    }

    /**
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = $this->container->get('nodetype_helper');

        $helper->setModuleSiteClass(NodeType::class);

        return $helper;
    }
}
