<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Cms\Controller\Admin\NodeOptionsController;
use Application\Core\Model\Module;
use Application\Core\Model\Permit;
use Application\Showcase\Model\NodeType;
use Pongho\Http\RedirectResponse;
use Pongho\Utilities\Inflector;

/**
 * Controller per la gestione delle opzioni dello showcase.
 */
class ShowcaseOptionsController extends NodeOptionsController
{
    /**
     * Azione 'showcase'.
     *
     * Gestisce le opzioni per il catalogo.
     *
     * @return \Pongho\Http\Response
     */
    public function showcaseAction()
    {
        $this->handleRequest([
            'enable_code'                    => 'boolean',
            'enable_price'                   => 'boolean',
            'enable_availabilities'          => 'boolean',
            'hide_unavailable_from_archives' => 'boolean',
            'enable_sizes'                   => 'boolean',
            'size_guide_node_type'           => 'integer',
        ]);

        if ($this->getRequest()->getMethod() === 'POST') {
            $site_module = $this->getSiteModule();

            if ($site_module->getOption('enable_sizes')) {
                if (Permit::findByKey('shop.admin.sizes') === null) {
                    Permit::create([
                        'key'        => 'shop.admin.sizes',
                        'is_enabled' => true,
                    ]);
                }

                return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/showcase/'));
            }
        }

        $language_id = $this->getHelper()->getLanguage()->language_id;

        /** @var \Application\Cms\Model\Manager\NodeTypeManager $node_type_manager */
        $node_type_manager = $this->getContainer()->get('node_type_manager');

        $size_guide_node_type = [];
        foreach ($node_type_manager->getNodeTypes($language_id) as $node_type) {
            $size_guide_node_type[$node_type->id] = $node_type->trans('name_plural', $language_id);
        }

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/nodes_options_showcase.php')
            ->assignVars([
                'action_form'           => $this->url('/' . $this->getParameter('path') . '/showcase/'),
                'disable_showcase_url'  => $this->getConvertToBaseUrl(),
                'size_guide_node_types' => $size_guide_node_type,
            ]);

        $this->displayTabs();

        return null;
    }

    /**
     * {@inheritdoc}
     */
    protected function getTabs()
    {
        $path = $this->getParameter('path');

        $tabs = parent::getTabs();
        $tabs[] = (object) [
            'name'      => $this->getHelper()->getTranslator()->trans('Showcase'),
            'link'      => $this->url("/{$path}/showcase/"),
            'is_active' => ($this->getAction() == 'showcase'),
        ];

        return $tabs;
    }

    /**
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = $this->container->get('nodetype_helper');

        $helper->setModuleSiteClass(NodeType::class);

        return $helper;
    }

    /**
     * @return string
     */
    private function getConvertToBaseUrl()
    {
        /** @var Module $module */
        $module = Module::find($this->getModuleId());
        $url = '/' . Inflector::pluralize($module->node_type) . '-showcase-converter/convert_to_base/';

        return $this->url($url);
    }
}
