<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Cms\Controller\Admin\NodesController;
use Application\Cms\Model\Node as CmsNode;
use Application\Core\Form\Field\DiscountField;
use Application\Core\Form\Field\PriceField;
use Application\Showcase\Model\Node;
use Application\Showcase\Model\Node as ShowcaseNode;
use Application\Showcase\Model\NodeType;
use Pongho\Core\TemplateResponse;
use Pongho\Form\Field\CheckboxField;
use Pongho\Form\Field\NumberField;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Field\TextField;
use Pongho\Form\Form;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Template\View;

/**
 * Controller per la gestione dei nodi dello showcase.
 */
class ShowcaseNodesController extends NodesController
{
    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Node::class;
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumnsBeforeTaxonomies(array $options)
    {
        $manual_order = $this->isManualOrder();

        $options[] = [
            'name'      => 'price',
            'label'     => $this->getHelper()->getTranslator()->trans('Price'),
            'class'     => 'price',
            'orderable' => !$manual_order,
            'render'    => function ($row) {
                return format_price($row->price);
            },
        ];

        return parent::getTableColumnsBeforeTaxonomies($options);
    }

    /**
     * {@inheritdoc}
     */
    public function renderTitleCell($row)
    {
        $code = parent::renderTitleCell($row);

        if ($row->code && $row->getSiteModule()->getOption('enable_code')) {
            $replace = '</strong> <span>('
                . $this->getHelper()->getTranslator()->trans('Code %code%', ['%code%' => $row->code])
                . ')</span><div class';
            $code = str_replace('</strong><div class', $replace, $code);
        }

        return $code;
    }

    /**
     * {@inheritdoc}
     */
    protected function getSearchFields()
    {
        $fields = parent::getSearchFields();

        if ($this->getHelper()->getModuleSite()->getOption('enable_code')) {
            $fields[] = 'code';
        }

        return $fields;
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelFieldsConfig()
    {
        $translator = $this->getHelper()->getTranslator();

        $options = parent::getModelFieldsConfig();
        if ($this->getHelper()->getModuleSite()->getOption('enable_code')) {
            $options['code'] = ['label' => $translator->trans('Code'), 'data-type' => 'string'];
        }

        return $options;
    }


    /**
     * {@inheritdoc}
     */
    protected function getDefaultFormsOptions()
    {
        $translator = $this->getHelper()->getTranslator();

        $form_config = parent::getDefaultFormsOptions();

        $module = $this->getHelper()->getNodeType();
        $site = $this->getHelper()->getSite();

        if ($module->getOption('enable_code')) {
            $form_config->addFieldBefore(
                'content/main/sidebar/status',
                'code',
                [
                    'class'      => TextField::class,
                    'label'      => $translator->trans('Code'),
                    'attributes' => ['class' => 'input_text'],
                ]
            );
        }

        if ($module->getOption('enable_price')) {
            $form_config->addFieldBefore(
                'content/main/sidebar/status',
                'price',
                [
                    'class'      => PriceField::class,
                    'label'      => $translator->trans('Price (format: %format%)', ['%format%' => format_price(1234567.89, false)]),
                    'attributes' => ['class' => 'input_text'],
                ]
            );

            if ($site->getOption('enable_offers')) {
                $form_config->addFieldBefore(
                    'content/main/sidebar/status',
                    'discount',
                    [
                        'class'      => DiscountField::class,
                        'label'      => $translator->trans('Discount (format: 20% or %format%)', ['%format%' => format_price(123.45, false)]),
                        'attributes' => ['class' => 'input_text'],
                    ]
                );
            }
        }

        /** @var \Application\Showcase\Discount\OrderDiscounts $orderDiscounts */
        $orderDiscounts = $this->getContainer()->get('shop_order_discounts');

        if ($orderDiscounts->isEnabled('order_amount')) {
            $form_config->addFieldBefore(
                'content/main/sidebar/status',
                'exclude_from_order_amount_discount',
                [
                    'class' => CheckboxField::class,
                    'label' => $translator->trans('Exclude from order amount discount'),
                ]
            );
        }

        if ($orderDiscounts->isEnabled('shopping_points') && $site->getOption('shopping_point_on_single_product')) {
            $form_config->addFieldBefore(
                'content/main/sidebar/status',
                'shopping_points',
                [
                    'class'      => NumberField::class,
                    'label'      => $translator->trans('Earnable points with this product'),
                    'attributes' => ['class' => 'input_text'],
                ]
            );
        }

        // Pannello taglie
        if ($module->getOption('enable_sizes')) {
            $form_config->addTab('content/sizes', $translator->trans('Sizes'), [
                'active' => $this->getActiveTab() === 'sizes',
            ]);

            // Tipo di taglia
            $form_config->addPanel('content/sizes/options');
            $form_config->addField(
                'content/sizes/options/size_type_id',
                [
                    'class'    => SelectField::class,
                    'label'    => $translator->trans('Size type'),
                    'options'  => $this->getSizeTypesSelectOptions(),
                    'settings' => [
                        'placeholder' => $translator->trans('Select a size type'),
                    ],
                ]
            );

            // Tabella taglie
            $form_config->addPanel('content/sizes/sizes-table', null, false, [
                'description' => '<p class="empty">' . $translator->trans('Select a size type') . '</p>',
            ]);
        }

        if ($site->getOption('enable_google_shopping')) {
            /** @var \Application\Showcase\GoogleShopping\Form\NodeFormConfigFilter $filter */
            $filter = $this->getContainer()->get('google_shopping_node_form_config_filter');
            $filter->filter($form_config);
        }

        return $form_config;
    }

    /**
     * {@inheritdoc}
     */
    protected function getSubject()
    {
        if ($this->subject === null) {
            $this->subject = parent::getSubject();

            if ($this->getHelper()->getNodeType()->getOption('enable_sizes')) {
                $this->subject->attachSetter(
                    'size_type_id',
                    function (ModelSubject $subject, $value): void {
                        if ($value === 'custom') {
                            $value = null;
                        }

                        /** @var \Application\Showcase\Model\Node $model */
                        $model = $subject->getModel();
                        $model->size_type_id = $value;
                    }
                );

                $this->subject->attachGetter(
                    'size_type_id',
                    function (ModelSubject $subject) {
                        /** @var \Application\Showcase\Model\Node $model */
                        $model = $subject->getModel();

                        if ($model->size_type_id !== null) {
                            return $model->size_type_id;
                        }

                        $sizes = $model->getSizes(true);

                        if (empty($sizes)) {
                            return null;
                        }

                        return 'custom';
                    }
                );
            }
        }

        return $this->subject;
    }

    /**
     * Restituisce le opzioni per la select dei tipi di taglia.
     *
     * @return array
     */
    protected function getSizeTypesSelectOptions()
    {
        $options = [
            'custom' => $this->getHelper()->getTranslator()->trans('Custom sizes'),
        ];

        /** @var $site_module \Application\Showcase\Model\NodeType */
        $site_module = $this->getHelper()->getNodeType();

        foreach ($site_module->getSizeTypes() as $type) {
            $options[$type->id] = $type->title;
        }

        return $options;
    }

    /**
     * Azione 'sizes'.
     *
     * Restituisce la lista delle taglie in base al loro tipo.
     *
     * @return \Pongho\Http\Response
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function sizesAction()
    {
        /** @var $model_class \Application\Showcase\Model\Node */
        $model_class = $this->getModelClass();

        if ($this->hasParameter('id')) {
            $node = $model_class::find($this->getParameter('id'));

            if ($node === null) {
                throw new HttpNotFoundException(sprintf('Node "%s" not found!', $this->getParameter('id')));
            }
        } else {
            $node = new $model_class($this->getModelBaseAttributes());
        }

        $size_type_id = $this->getRequest()->query->get('size_type');

        if ($size_type_id === 'custom') {
            $node->size_type_id = null;

            return new TemplateResponse($this->getCustomSizesView($node));
        }

        if ($size_type_id === null) {
            throw new HttpNotFoundException('Size type is not specified!');
        }

        $size_type_id = intval($size_type_id);

        $node->size_type_id = $size_type_id;

        return new TemplateResponse($this->getSizesView($node));
    }

    /**
     * Callback 'before_set_template'.
     *
     * @throws \RuntimeException
     */
    protected function beforeSetTemplateCallback(CmsNode $node)
    {
        if (!($node instanceof ShowcaseNode)) {
            throw new \RuntimeException(
                sprintf('Parameter $node in %s must be instance of %s!', __METHOD__, Node::class)
            );
        }

        parent::beforeSetTemplateCallback($node);
    }

    /**
     * {@inheritdoc}
     */
    protected function renderAddEdit(Form $form)
    {
        parent::renderAddEdit($form);

        $this->displaySizes();
    }

    /**
     * Callback 'after_save'.
     *
     * @throws \RuntimeException
     */
    protected function afterSaveCallback(Form $form)
    {
        /** @var \Application\Showcase\Model\Node $node */
        $node = $form->getSubject()->getModel();

        if (!($node instanceof ShowcaseNode)) {
            throw new \RuntimeException(
                sprintf('Parameter $node in %s must be instance of %s!', __METHOD__, Node::class)
            );
        }

        parent::afterSaveCallback($form);

        $this->saveSizes($node);
    }

    /**
     * Visualizza l’elenco delle taglie.
     */
    protected function displaySizes()
    {
        if (!$this->getHelper()->getNodeType()->getOption('enable_sizes')) {
            return;
        }

        /** @var \Application\Showcase\Model\Node $node */
        $node = $this->getModel();

        $options = [
            'url' => $this->url('/' . $this->getParameter('path') . '/sizes/' . ($this->getAction() === 'edit' ? "{$node->id}/" : ''), '&'),
        ];

        $this->getHelper()
            ->addJavascriptInline('var sizes = ' . json_encode($options) . ';')
            ->addJavascript(pongho_url(
                '/Application/Showcase/Resources/views/js/jquery.nodes-sizes.js?v='
                . filemtime(PONGHO_PATH . '/Application/Showcase/Resources/views/js/jquery.nodes-sizes.js')
            ));
    }

    /**
     * Resituisce la vista delle taglie.
     *
     * @return \Pongho\Template\View;
     */
    protected function getSizesView(ShowcaseNode $node)
    {
        $view = new View(__DIR__ . '/../../Resources/views/nodes_sizes.php');
        $view
            ->assignVars($this->getHelper()->getViewGlobalVariables())
            ->assignVars([
                'availabilities_enabled'           => $this->getHelper()->getNodeType()->getOption('enable_availabilities'),
                'gs_enabled'                       => $this->getHelper()->getSite()->getOption('enable_google_shopping'),
                'shopping_point_on_single_product' => $this->getHelper()->getSite()->getOption('shopping_point_on_single_product'),
                'sizes'                            => $node->getSizes(true),
            ]);

        return $view;
    }

    /**
     * Resituisce la vista per le taglie personalizzate.
     *
     * @return \Pongho\Template\View;
     */
    protected function getCustomSizesView(ShowcaseNode $node)
    {
        $view = new View(__DIR__ . '/../../Resources/views/nodes_sizes_custom.php');
        $view
            ->assignVars($this->getHelper()->getViewGlobalVariables())
            ->assignVars([
                'availabilities_enabled'           => $this->getHelper()->getNodeType()->getOption('enable_availabilities'),
                'gs_enabled'                       => $this->getHelper()->getSite()->getOption('enable_google_shopping'),
                'shopping_point_on_single_product' => $this->getHelper()->getSite()->getOption('shopping_point_on_single_product'),
                'sizes'                            => $node->getSizes(true),
            ]);

        return $view;
    }

    /**
     * Salva le taglie.
     */
    protected function saveSizes(ShowcaseNode $node)
    {
        if (!$this->getHelper()->getNodeType()->getOption('enable_sizes')) {
            return;
        }

        // taglie passate via POST
        $post_sizes = $this->getRequest()->post->get('sizes', []);
        $is_custom = $this->getRequest()->post->get('node[size_type_id]', null, true) === 'custom';

        foreach ($post_sizes as &$attributes) {
            if ($is_custom) {
                $attributes['name'] = isset($attributes['name']) ? html_escape(trim($attributes['name'])) : '';
            }

            $attributes['price'] = (isset($attributes['price']) ? parse_price($attributes['price']) : 0) ?: $node->price;
            $attributes['availabilities'] = isset($attributes['availabilities']) ? intval($attributes['availabilities']) : 0;
            $attributes['shopping_points'] = isset($attributes['shopping_points']) ? intval($attributes['shopping_points']) : 0;
            $attributes['gtin'] ??= '';
            $attributes['mpn'] ??= '';
            $attributes['gs_enabled'] = isset($attributes['gs_enabled']) && $attributes['gs_enabled'];
        }

        if ($is_custom) {
            $node->setCustomSizes($post_sizes);
        } else {
            $node->setSizes($post_sizes);
        }

        $node->save();
    }

    /**
     * Normalizza un numero assicurandosi che sia un valore float valido.
     *
     * @param string $value
     * @return float
     */
    protected function validateFloat($value)
    {
        return floatval(str_replace(',', '.', $value));
    }

    /**
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = $this->container->get('nodetype_helper');

        $helper->setModuleSiteClass(NodeType::class);

        return $helper;
    }
}
