<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Admin\Form\PanelFieldset;
use Application\Core\Form\Field\PriceField;
use Application\Showcase\Form\Field\PaymentMethodOptionsFieldset;
use Application\Showcase\Model\NodeType;
use Application\Showcase\Model\Payment;
use Application\Showcase\Payment as PaymentHandler;
use Pongho\Form\Field\CheckboxField;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Field\TextareaField;
use Pongho\Form\Field\TextField;
use Pongho\Form\Fieldset;
use Pongho\Form\Form;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpUnauthorizedException;

/**
 * Controller per la gestione dei metodi di pagamento in admin.
 */
class PaymentsController extends CrudFormController
{
    /**
     * @var int
     */
    private $rowCounter = 1;

    /**
     * @var PaymentHandler\BasePayment[]|string[]
     */
    private $paymentMethodHandlers;

    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        parent::boot();

        // Potrei aver impostato la risposta con il redirect alla pagina di login.
        if ($this->response) {
            return;
        }

        // Permesso consentito?
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.payments.manage')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'shop.payment_methods';
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Payment::class;
    }

    /**
     * {@inheritdoc}
     */
    protected function canAdd()
    {
        return $this->getHelper()->userHasPermit('shop.admin.payments.add');
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelOptions()
    {
        return [
            'order' => 'position ASC',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function displayArchive()
    {
        $this->rowCounter = 1;

        return parent::displayArchive();
    }

    /**
     * {@inheritdoc}
     *
     * @param Payment $row
     */
    public function parseArchiveRow($row)
    {
        $path = $this->getParameter('path');
        $translator = $this->getHelper()->getTranslator();

        $actions = [];
        $orderActions = [];

        // Edit
        $actions[] = [
            $translator->trans('Edit'),
            'href'  => $this->url("/{$path}/edit/{$row->id}/"),
            'class' => 'edit',
        ];

        // Delete
        if ($this->getHelper()->userHasPermit('shop.admin.payments.delete')) {
            $actions[] = [
                $translator->trans('Delete'),
                'href'  => $this->url("/{$path}/delete/{$row->id}/"),
                'class' => 'delete',
            ];
        }

        // Up
        if ($this->rowCounter !== 1) {
            $orderActions[] = [
                $translator->trans('Up'),
                'href'  => $this->url("/{$path}/moveup/{$row->id}/"),
                'title' => $translator->trans('Move up'),
                'class' => 'move-up',
            ];
        }

        // Down
        if ($this->rowCounter !== $this->getArchiveManager()->getTotalRecords()) {
            $orderActions[] = [
                $translator->trans('Down'),
                'href'  => $this->url("/{$path}/movedown/{$row->id}/"),
                'title' => $translator->trans('Move down'),
                'class' => 'move-down',
            ];
        }

        $row->actions = $this->parseActions($actions);
        $row->order_actions = $this->parseActions($orderActions);

        $this->rowCounter++;
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        $translator = $this->getHelper()->getTranslator();

        return [
            [
                'name'   => 'order',
                'label'  => $translator->trans('Order'),
                'class'  => 'order',
                'render' => function ($row) {
                    return $row->order_actions;
                },
            ],
            [
                'name'   => 'name',
                'label'  => $translator->trans('Payment'),
                'class'  => 'main',
                'render' => function (Payment $row) {
                    return "<strong>{$row->name}</strong><div class=\"actions\"><span>{$row->actions}</span></div>";
                },
            ],
            [
                'name'   => 'is_enabled',
                'label'  => $translator->trans('Enabled'),
                'render' => function (Payment $row) use ($translator) {
                    return $row->is_enabled ? $translator->trans('Yes') : $translator->trans('No');
                },
            ],
        ];
    }

    /**
     * Restituisce l’elenco delle azioni di massa.
     *
     * @return array
     */
    protected function getMassActions()
    {
        $actions = [];

        if ($this->getHelper()->getUser()->hasPermit('shop.admin.payments.delete')) {
            $actions['delete'] = $this->getHelper()->getTranslator()->trans('Delete');
        }

        return $actions;
    }

    /**
     * Azione `moveup`.
     *
     * Permette di spostare in alto il pagamento.
     *
     * @return \Pongho\Http\Response
     */
    public function moveupAction()
    {
        /** @var Payment $payment */
        $payment = Payment::find($this->getParameter('id', null, true));

        return $this->movePayment($payment, -15);
    }

    /**
     * Azione `movedown`.
     *
     * Permette di spostare in alto il pagamento.
     *
     * @return \Pongho\Http\Response
     */
    public function movedownAction()
    {
        /** @var Payment $payment */
        $payment = Payment::find($this->getParameter('id', null, true));

        return $this->movePayment($payment, 15);
    }

    /**
     * Permette lo spostamento del pagamento.
     *
     * @param int                                 $direction
     * @return \Pongho\Http\Response
     */
    protected function movePayment(Payment $payment, $direction)
    {
        $payment->position += $direction;
        $payment->save();

        $position = 10;

        /** @var Payment $payment */
        foreach (Payment::all(['order' => 'position ASC']) as $payment) {
            $payment->position = $position;
            $payment->save();

            $position += 10;
        }

        return $this->getHelper()->redirectResponse($this->url('/' . $this->getParameter('path') . '/'));
    }

    /**
     * {@inheritdoc}
     */
    protected function renderAddEdit(Form $form)
    {
        parent::renderAddEdit($form);

        $this->getHelper()->addJavascriptInline(
            <<<JS
(function () {
    "use strict";

    function showPayment() {
        var select = $('#field-payment-method-handler_class');
        var payments = $('.payment-method');
        var option = select.find('option:selected');
        var payment = $('#fieldset-payment-method-' + option.data('target') + '-options');

        payments.hide();
        payment.show();
    }

    $(document.body).on('change', '#field-payment-method-handler_class', showPayment);
    $('.form-wrapper').on('after-update', showPayment);
}());
JS
        );
    }

    /**
     * {@inheritdoc}
     *
     * @param Payment $payment
     */
    public function getAddEditTitle($payment)
    {
        if ($this->getAction() === 'add') {
            return $this->getHelper()->getTranslator()->trans('Add payment method');
        }

        return $this->getHelper()->getTranslator()->trans(
            'Edit the payment method “%title%”',
            [
                '%title%' => $payment->name,
            ]
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function checkAddPermit()
    {
        if (!$this->canAdd()) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function checkEditPermit()
    {
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.payments.manage')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function checkDeletePermit()
    {
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.payments.delete')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        return $this->getHelper()->filter(
            $this,
            $this->getEventNamespace('filter_form_options'),
            $this->getDefaultFormsOptions(),
            [
                'action' => $this->getAction(),
            ]
        );
    }

    /**
     * @return FormConfig
     */
    protected function getDefaultFormsOptions()
    {
        /** @var Payment $paymentModel */
        $paymentModel = $this->getModel();

        if ($this->action == 'add') {
            $paymentModel->is_enabled = true;
        }

        $subject = new ModelSubject($paymentModel);

        $translator = $this->getHelper()->getTranslator();
        $paymentMethodHandlers = $this->getPaymentMethodHandlers();

        $config = new FormConfig('payment-method', $subject, $this->getHelper()->getLocalization());
        $config->addBaseStructure($this->getActiveTab(), $this->currentUrl());

        $config->addField(
            'content/main',
            [
                'class' => PanelFieldset::class,
                'label' => $translator->trans('Main'),
            ]
        );

        $config->addField(
            'content/main/payment',
            [
                'class'      => Fieldset::class,
                'attributes' => ['class' => 'panel'],
                'settings'   => [
                    'hide_label' => true,
                ],
            ]
        );

        $config->addField(
            'content/main/payment/name',
            [
                'class' => TextField::class,
                'label' => $translator->trans('Name'),
            ]
        );

        if ($this->getHelper()->getUser()->isFounder()) {
            $config->addField(
                'content/main/payment/handler_class',
                [
                    'class'   => SelectField::class,
                    'label'   => $translator->trans('Handler'),
                    'options' => function () use ($paymentMethodHandlers) {
                        $options = [];
                        foreach ($paymentMethodHandlers as $handlerName => $className) {
                            $options[$className] = [$className::getName(), 'data-target' => $handlerName];
                        }

                        return $options;
                    },
                ]
            );
        }

        $config->addFields(
            'content/main/payment',
            [
                'checkout_description' => [
                    'class'      => TextareaField::class,
                    'label'      => $translator->trans('Checkout description'),
                    'attributes' => [
                        'rows' => 5,
                    ],
                ],
                'email_description'    => [
                    'class'      => TextareaField::class,
                    'label'      => $translator->trans('Email description'),
                    'attributes' => [
                        'rows' => 5,
                    ],
                ],
                'cost'                 => [
                    'class' => PriceField::class,
                    'label' => $translator->trans('Cost'),
                ],
                'is_enabled'           => [
                    'class'       => CheckboxField::class,
                    'label'       => $translator->trans('Enabled'),
                    'description' => $translator->trans('Enable this payment method'),
                ],
            ]
        );

        $paymentOptions = $paymentModel->getHandlerOptions();
        if ($this->getHelper()->getUser()->isFounder()) {
            foreach ($paymentMethodHandlers as $className) {
                if ($paymentModel->handler_class === $className) {
                    $fields = $className::getFormFieldsConfig($config, $translator, $paymentOptions);
                } else {
                    $fields = $className::getFormFieldsConfig($config, $translator);
                }

                if ($fields) {
                    $config->addField(
                        'content/main/' . $className::getName() . '-options',
                        [
                            'class'      => PaymentMethodOptionsFieldset::class,
                            'attributes' => [
                                'class' => 'payment-method panel',
                                'style' => $paymentModel->handler_class === $className ? '' : 'display: none;',
                            ],
                            'label'      => $className::getName(),
                        ]
                    );

                    $config->addFields(
                        'content/main/' . $className::getName() . '-options',
                        $fields
                    );
                }
            }
        } else {
            /** @var string|\Application\Showcase\Payment\BasePayment $className */
            $className = $paymentModel->handler_class;
            $fields = $className::getFormFieldsConfig($config, $translator, $paymentOptions);

            if ($fields) {
                $config->addField(
                    'content/main/' . $className::getName() . '-options',
                    [
                        'class'      => PaymentMethodOptionsFieldset::class,
                        'attributes' => ['class' => 'panel'],
                        'label'      => $className::getName(),
                    ]
                );

                $config->addFields(
                    'content/main/' . $className::getName() . '-options',
                    $className::getFormFieldsConfig($config, $translator, $paymentOptions)
                );
            }
        }

        return $config;
    }

    /**
     * @return PaymentHandler\BasePayment[]|string[]
     */
    private function getPaymentMethodHandlers()
    {
        if ($this->paymentMethodHandlers === null) {
            /** @var PaymentHandler\BasePayment[]|string[] $classes */
            $classes = $this->getHelper()->filter(
                $this,
                $this->getEventNamespace('filter_payments_classes'),
                [
                    PaymentHandler\Simple\Payment::class,
                    PaymentHandler\ZeroOrder\Payment::class,
                    PaymentHandler\ConsorzioTriveneto\Payment::class,
                    PaymentHandler\PagOnline\Payment::class,
                    PaymentHandler\PayPal\Payment::class,
                    PaymentHandler\Soisy\Payment::class,
                    PaymentHandler\Unicredit\Payment::class,
                    PaymentHandler\VirtualPay\Payment::class,
                    PaymentHandler\Volksbank\Payment::class,
                ]
            );

            $this->paymentMethodHandlers = [];
            foreach ($classes as $class) {
                $this->paymentMethodHandlers[strtolower((string) $class::getName())] = $class;
            }
        }

        return $this->paymentMethodHandlers;
    }

    /**
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = $this->container->get('nodetype_helper');

        $helper->setModuleSiteClass(NodeType::class);

        return $helper;
    }
}
