<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin;

use Application\Admin\Controller\AdminController;
use Application\Admin\Form\FormConfig;
use Application\Admin\Form\Generator\AdminGenerator;
use Application\Core\Form\Field\DiscountField;
use Application\Showcase\Model\OrderDiscount;
use Application\Showcase\Model\OrderDiscountCondition;
use Pongho\Form\Form;
use Pongho\Form\Repeater\ModelRepeater;
use Pongho\Form\Subject\ModelRowSubject;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Form\Utilities\FormBuilder;
use Pongho\Http\Exception\HttpUnauthorizedException;

class OrdersAmountDiscountController extends AdminController
{
    /**
     * {@inheritdoc}
     *
     * TODO: Valutare se questo metodo serve...
     */
    protected function boot()
    {
        parent::boot();

        // Potrei aver impostato la risposta con il redirect alla pagina di login.
        if ($this->response) {
            return;
        }

        // Permesso consentito?
        if (!$this->getHelper()->getUser()->hasPermit('shop.admin.orders_amount_discount')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * Azione `index`.
     *
     * @return \Pongho\Http\Response
     */
    public function indexAction()
    {
        $translator = $this->getHelper()->getTranslator();

        $form = $this->getForm();
        $form->handleRequest($this->getRequest());

        if ($form->hasHandled() && !$form->hasErrors()) {
            $path = $this->getParameter('path');

            return $this->getHelper()->redirectResponse($this->url('/' . $path . '/', '&'));
        }

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../../Admin/Resources/views/edit.php')
            ->assignVars([
                'title'   => $translator->trans('Management of discounts based on the amount of the order'),
                'content' => $form,
            ]);

        // Evito di salvare la form utilizzando ajax.
        $this->getHelper()
            ->addJavascriptInline('Pongho.action = "add";')
            ->addJavascript(pongho_url(
                '/Application/Admin/Resources/views/js/plugins.js?v='
                . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/js/plugins.js')
            ))
            ->addJavascript(pongho_url(
                '/Application/Admin/Resources/views/js/scripts.js?v='
                . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/js/scripts.js')
            ));

        return null;
    }

    /**
     * @return Form
     */
    protected function getForm()
    {
        static $form;

        if ($form === null) {
            /** @var \Pongho\Form\Form $form */
            $form = FormBuilder::buildConfig($this->getFormConfig());

            $form->setGenerator(new AdminGenerator($this->getHelper()->getLocalization()));
            $form->setAction($this->currentUrl());
        }

        return $form;
    }

    /**
     * @return FormConfig
     */
    protected function getFormConfig()
    {
        $translator = $this->getHelper()->getTranslator();

        $config = new FormConfig('discount', $this->getSubject(), $this->getHelper()->getLocalization());

        // Struttura
        $config->addBaseStructure('main', $this->currentUrl());
        $config->addTab('content/main', $translator->trans('Main'));
        $config->addPanel('content/main/discount');

        $config->addFields(
            'content/main/discount',
            [
                'conditions' => [
                    'class'    => ModelRepeater::class,
                    'settings' => [
                        'filter_model' => $this->filterConditionsModel(...),
                        'row_settings' => [
                            'delete-label' => $translator->trans('Delete'),
                            'hide-open'    => true,
                            'show-delete'  => true,
                        ],
                        //'view'         => new ViewClosure(array($this, 'repeaterView')),
                    ],
                ],
            ]
        );

        $config->addRepeaterRow('content/main/discount/conditions');

        $config->addFields(
            'content/main/discount/conditions/main',
            [
                'value'     => [
                    'class'       => DiscountField::class,
                    'label'       => $translator->trans('Discount'),
                    'description' => $translator->trans('Discount (format: 20% or %format%)', ['%format%' => format_price(123.45, false)]),
                ],
                'condition' => [
                    'class' => \Application\Core\Form\Field\PriceField::class,
                    'label' => $translator->trans('Min. price'),
                ],
            ]
        );

        return $config;
    }

    /**
     * @return ModelSubject
     */
    protected function getSubject()
    {
        static $subject;

        if ($subject === null) {
            $model = $this->getModel();
            $subject = new ModelSubject($model);

            $subject->attachGetter('conditions', $this->subjectConditionGetter(...));
        }

        return $subject;
    }

    /**
     * @return OrderDiscount
     */
    protected function getModel()
    {
        static $model;

        if ($model === null) {
            $options = [
                'conditions' => ['handler_name = :handler', 'handler' => 'order_amount'],
            ];

            $model = OrderDiscount::first($options);

            if ($model === null) {
                $model = new OrderDiscount([
                    'handler_name' => 'order_amount',
                ]);
            }
        }

        return $model;
    }

    /**
     * @return ModelRowSubject[]
     */
    public function subjectConditionGetter(ModelSubject $subject)
    {
        $options = [
            'conditions' => ['discount_id = :discount', 'discount' => $subject->get('id')],
        ];

        /** @var OrderDiscountCondition $condition */
        $conditions = [];
        foreach (OrderDiscountCondition::all($options) as $condition) {
            $conditions[$condition->id] = new ModelRowSubject($condition);
        }

        return $conditions;
    }

    /**
     * @param string        $key
     * @return OrderDiscountCondition
     */
    public function filterConditionsModel($key, array $request, ModelRepeater $repeater)
    {
        if (is_numeric($key)) {
            return OrderDiscountCondition::find($key);
        } else {
            return new OrderDiscountCondition([
                'discount_id' => $repeater->getSubject()->get('id'),
            ]);
        }
    }

    /**
     * @return string
     */
    public function repeaterView()
    {
    }
}
