<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Showcase\Controller\Admin\Orders\EditPanel;

use Application\Admin\Form\Generator\AdminGenerator;
use Application\Admin\Utilities\PonghoUrl;
use Application\Cms\Utilities\NodeTypeHelper;
use Application\Core\I18n\Translation\Translator;
use Application\Showcase\Controller\Admin\Orders\OrderEditPanelInterface;
use Application\Showcase\Domain\Order\Command\ConfirmOrderPaymentCommandFactory;
use Application\Showcase\Model\Order;
use Pongho\Form\Field\ButtonField;
use Pongho\Form\Field\DateField;
use Pongho\Form\Field\HiddenField;
use Pongho\Form\Fieldset;
use Pongho\Form\Form;
use Pongho\Form\Subject\ArrayRowSubject;
use Pongho\Http\Request;
use Psr\Log\LoggerInterface;

/**
 * ConfirmPayment
 */
class ConfirmOrderPaymentPanel implements OrderEditPanelInterface
{
    /**
     * @var Translator
     */
    private $translator;

    /**
     * @var PonghoUrl
     */
    private $ponghoUrl;

    /**
     * @var NodeTypeHelper
     */
    private $helper;

    /**
     * @var ConfirmOrderPaymentCommandFactory
     */
    private $commandFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    public function __construct(
        Translator $translator,
        PonghoUrl $ponghoUrl,
        NodeTypeHelper $helper,
        ConfirmOrderPaymentCommandFactory $commandFactory,
        LoggerInterface $logger
    ) {
        $this->translator = $translator;
        $this->ponghoUrl = $ponghoUrl;
        $this->helper = $helper;
        $this->commandFactory = $commandFactory;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return 'confirm-payment';
    }

    /**
     * {@inheritdoc}
     */
    public function getLabel()
    {
        return $this->translator->trans('Confirm payment');
    }

    /**
     * {@inheritdoc}
     */
    public function renderPanel(Order $order)
    {
        return $this->buildForm($order)->render();
    }

    /**
     * {@inheritdoc}
     *
     * @todo: Passare per la Form.
     */
    public function handleRequest(Order $order, Request $request)
    {
        $result = $this->commandFactory->createFromRequest($order, $request)->handle();
        $resultArray = [
            'message' => $result->isSuccessful() ? $result->getMessage() : '',
            'alerts'  => $result->getAlerts(),
            'errors'  => $result->getErrors(),
        ];

        $this->logger->info('[SHOP] Order Payment Confirm Result', [
            'order'  => $order->getId(),
            'result' => $resultArray,
        ]);

        $this->helper->getBodyView()->assignVars($resultArray);
    }

    /**
     * @return Form
     */
    protected function buildForm(Order $order)
    {
        $action = $this->ponghoUrl->url('/shop-orders/view/' . $order->getId() . '/');

        return (new Form('order', $this->buildFormSubject(), $action, 'post', new AdminGenerator()))
            ->addCssClass('pongho-form')
            ->add(
                (new HiddenField('action'))
            )
            ->add(
                (new Fieldset('data', false))
                    ->setSetting('hide_label', true)
                    ->add(
                        (new DateField('paid_at'))
                            ->setLabel($this->translator->trans('Payment date'))
                            ->addCssClass('pongho-field input_date')
                    )
            )
            ->add(
                (new ButtonField('confirm-payment'))
                    ->setLabel($this->translator->trans('Confirm'))
                    ->setSetting('type', 'submit')
                    ->addCssClass('pongho-button primary')
            )
            ->add(
                (new ButtonField('cancel'))
                    ->setLabel($this->translator->trans('Cancel'))
                    ->setSetting('type', 'button')
                    ->addCssClass('pongho-button cancel')
            );
    }

    /**
     * @return ArrayRowSubject
     */
    protected function buildFormSubject()
    {
        return new ArrayRowSubject([
            'action'  => $this->getId(),
            'paid_at' => date('Y-m-d'),
        ]);
    }
}
