<?php

use Application\Core\Entity\ImageInterface;
use Application\Core\I18n\LocaleInterface;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Platforms\PostgreSQLPlatform;
use Pongho\Core\Kernel;

/**
 * Genera l’url relativo.
 *
 * Esempio:
 * <code>
 *   echo url('/user/login/'); // Stampa /user/login/
 *   echo url('/'); // Stampa /
 *   echo url(''); // Stampa /
 * </code>
 */
function url(string $url = '/'): string
{
    static $site_path;

    if (is_null($site_path)) {
        /** @var \Pongho\Http\Request $request */
        $request = Kernel::instance()->getContainer()->get('request');

        $site_path = $request->getBasePath();
    }

    return prepend_url($url, $site_path);
}

/**
 * Genera l’url relativo aggiungendo il percorso della lingua.
 *
 * Ad esempio, se sono nella sezione inglese (percorso '/en/'):
 * <code>
 *   echo lang_url('/user/login/'); // Stampa /en/user/login/
 *   echo lang_url('/'); // Stampa /en/
 *   echo lang_url(''); // Stampa /en/
 * </code>
 */
function lang_url(string $url = '/'): string
{
    static $lang_path;

    if (is_null($lang_path)) {
        $lang_path = Kernel::instance()->getContainer()->get('language')->getLanguagePath();
    }

    return url($lang_path . ltrim($url, '/'));
}

/**
 * Genera l’url assoluto.
 *
 * Questa funzione è identica ad url(), con la differenza che aggiunge tutto il percorso del sito comprensivo di dominio.
 *
 * Esempio:
 * <code>
 *   echo absolute_url('/user/login/'); // Stampa http://www.example.com/user/login/
 *   echo absolute_url('/'); // Stampa http://www.example.com/
 *   echo absolute_url(''); // Stampa http://www.example.com/
 * </code>
 */
function absolute_url(string $url = '/'): string
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return prepend_url($url, $site->getDomain());
}

/**
 * Genera l’url assoluto aggiungendo il percorso della lingua.
 *
 * Ad esempio, se sono nella sezione inglese (percorso '/en/'):
 * <code>
 *   echo absolute_lang_url('/user/login/'); // Stampa http://www.example.com/en/user/login/
 *   echo absolute_lang_url('/'); // Stampa http://www.example.com/en/
 *   echo absolute_lang_url(''); // Stampa http://www.example.com/en/
 * </code>
 */
function absolute_lang_url(string $url = '/'): string
{
    static $lang_path;

    if (is_null($lang_path)) {
        $lang_path = Kernel::instance()->getContainer()->get('language')->getLanguagePath();
    }

    return absolute_url($lang_path . ltrim($url, '/'));
}

/**
 * Genera l’url assoluto per un percorso del tema.
 */
function themes_url(string $url): string
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return absolute_url($site->getThemesUrl($url));
}

/**
 * Genera l’url assoluto per una immagine.
 */
function images_url(string $url): string
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return absolute_url($site->getImagesUrl($url));
}

/**
 * Genera l’url assoluto per un file caricato.
 */
function uploads_url(string $url): string
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return absolute_url($site->getUploadsUrl($url));
}

/**
 * Genera l’url assoluto per i file nella cartella di Pongho.
 */
function pongho_url(string $url): string
{
    $dic = Kernel::instance()->getContainer();

    /** @var \Application\Core\Model\Site $site */
    $site = $dic->get('main_site');

    $path = $site->getPermalink() . trim((string) $dic->getParameter('pongho_path', 'pongho'), '/');

    return $path . '/' . ltrim($url, '/');
}

/**
 * Restituisce il percorso di una immagine permettendo di specificare dimensioni e tipo di taglio.
 */
function src(?ImageInterface $image = null, string $image_size = 'main', ?string $image_not_found = null): string
{
    if (!$image instanceof ImageInterface || !file_exists($image->filepath())) {
        if ($image_not_found) {
            return $image_not_found;
        }

        return themes_url('/images/not-found-' . $image_size . '.png');
    }

    return $image->getSizePermalink($image_size);
}

/**
 * Aggiunge un percorso all'inizio dell'url.
 *
 * @access private
 */
function prepend_url(string $url, string $prepend): string
{
    if (preg_match('#^((f|ht)tp(s)?://|mailto:|callto:|\#)#', $url)) {
        return $url;
    }

    return $prepend . '/' . ltrim($url, '/');
}

/**
 * Restituisce l’url relativo corrente.
 */
function current_url(): string
{
    /** @var \Pongho\Http\Request $request */
    $request = Kernel::instance()->getContainer()->get('request');

    return $request->getPathInfo();
}

/**
 * Indica se sono in home page.
 */
function is_home(): bool
{
    $container = Kernel::instance()->getContainer();

    /**
     * @var \Pongho\Http\Request                 $request
     * @var \Application\Core\Model\LanguageSite $language
     */
    $request = $container->get('request');
    $language = $container->get('language_site');

    return $request->getPathInfo() === $language->getPath();
}

/**
 * Indica se sono in amministrazione.
 */
function is_pongho(): bool
{
    /** @var \Pongho\Http\Request $request */
    $request = Kernel::instance()->getContainer()->get('request');

    return str_starts_with($request->getPathInfo(), '/pongho/');
}

/**
 * Indica se la richiesta è di tipo AJAX.
 */
function is_ajax(): bool
{
    /** @var \Pongho\Http\Request $request */
    $request = Kernel::instance()->getContainer()->get('request');

    return $request->isAjax();
}

/**
 * Prepara la password per i dovuti controlli.
 */
function clean_password(string $password): string
{
    $password = trim($password);

    if ($password === '') {
        return '';
    }

    return md5($password);
}

/**
 * @access private
 */
function _get_locale(): LocaleInterface
{
    static $locale;

    if (!$locale) {
        /**
         * @var \Application\Core\Model\LanguageSite  $languageSite
         * @var \Application\Core\I18n\LocaleRegistry $localeRegistry
         */
        $languageSite = Kernel::instance()->getContainer()->get('language_site');
        $localeRegistry = Kernel::instance()->getContainer()->get('i18n.locale_registry');

        $locale = $localeRegistry->getLocale($languageSite->getLanguage()->getCulture());
    }

    return $locale;
}

/**
 * Restituisce un prezzo formattato in base alla localizzazione corrente.
 *
 * @param int|float $price
 * @return string
 */
function format_price($price, bool $showCurrency = true, bool $showDecimal = true)
{
    return _get_locale()->formatPrice($price, $showCurrency, $showDecimal);
}

/**
 * Converte un prezzo da formattato a float in base alla localizzazione corrente.
 */
function parse_price(string $price): float
{
    return _get_locale()->parsePrice($price);
}

/**
 * Restituisce il nome del mese in base al numero.
 *
 * <code>
 *   echo get_month(1); // Gennaio
 * </code>
 *
 * @param mixed $n
 * @return mixed|string
 */
function get_month($n)
{
    $n = intval($n);

    $value = "MON_$n";
    if (function_exists('nl_langinfo')) {
        return nl_langinfo(constant($value));
    } else {
        $language = Kernel::instance()->getContainer()->get('language')->getLanguage();

        require_once PONGHO_PATH . "/src/languages/{$language->iso}/functions.php";

        $function_name = "Languages\\{$language->iso}\\nl_langinfo";
        if (function_exists($function_name)) {
            return call_user_func($function_name, $value);
        } else {
            // I sistemi windows non hanno la funzione nl_langinfo, restituisco il numero del mese
            return $value;
        }
    }
}

/**
 * Esegue gli shortcode al testo passato.
 */
function do_shortcode(string $text): string
{
    return Kernel::instance()->getContainer()->get('shortcode')->parse($text);
}

/**
 * Restituisce il link per il login con LinkedIn.
 */
function get_linkedin_auth_url(): string
{
    return url('/user/linkedin-auth/');
}

/**
 * Restituisce l’ID dell’ultima riga inserita in una tabella.
 *
 * La query di INSERT deve essere appena stata fatta.
 *
 * @link https://github.com/doctrine/dbal/issues/4687
 *
 * @throws \Doctrine\DBAL\Exception
 */
function dbal_last_insert_id(Connection $connection, ?string $sequence_name = null): int
{
    if ($connection->getDatabasePlatform() instanceof PostgreSQLPlatform) {
        // La soluzione ufficiale è di ottenere un ID prima dell’insert, ma nel caso di
        // PostgreSQL si possono usare le sequence in sicurezza perché sono per sessione.
        // @link https://www.postgresql.org/docs/current/functions-sequence.html
        return (int) $connection->fetchOne("SELECT CURRVAL('$sequence_name')");
    }

    // Nel caso di MySQL posso usare il metodo originale
    return (int) $connection->lastInsertId();
}
