<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Utilities;

class Validator
{
    /**
     * @param string $code
     * @return bool
     *
     * @link http://www.icosaedro.it/cf-pi/
     */
    public static function codice_fiscale($code)
    {
        $code = trim((string) $code);

        if ($code === '') {
            return true;
        }

        $code = strtoupper($code);

        // Per le aziende sprovviste di codice fiscale, viene usata la partita iva
        if (self::partita_iva($code)) {
            return true;
        }

        // Se non si ha corrispondenza con la partita iva, faccio controlli sul codice fiscale

        // Lunghezza
        if (strlen($code) != 16) {
            return false;
        }

        // Sintassi
        $regex = '#^([A-Z]{6})([A-Z0-9]{2})([ABCDEHLMPRST])([A-Z0-9]{2})([A-Z])([A-Z0-9]{3})([A-Z])$#';
        if (!preg_match($regex, $code)) {
            return false;
        }

        // Codice finale di controllo
        $odd = [
            '0' => 1,
            '1' => 0,
            '2' => 5,
            '3' => 7,
            '4' => 9,
            '5' => 13,
            '6' => 15,
            '7' => 17,
            '8' => 19,
            '9' => 21,
            'A' => 1,
            'B' => 0,
            'C' => 5,
            'D' => 7,
            'E' => 9,
            'F' => 13,
            'G' => 15,
            'H' => 17,
            'I' => 19,
            'J' => 21,
            'K' => 2,
            'L' => 4,
            'M' => 18,
            'N' => 20,
            'O' => 11,
            'P' => 3,
            'Q' => 6,
            'R' => 8,
            'S' => 12,
            'T' => 14,
            'U' => 16,
            'V' => 10,
            'W' => 22,
            'X' => 25,
            'Y' => 24,
            'Z' => 23,
        ];

        $even = [
            '0' => 0,
            '1' => 1,
            '2' => 2,
            '3' => 3,
            '4' => 4,
            '5' => 5,
            '6' => 6,
            '7' => 7,
            '8' => 8,
            '9' => 9,
            'A' => 0,
            'B' => 1,
            'C' => 2,
            'D' => 3,
            'E' => 4,
            'F' => 5,
            'G' => 6,
            'H' => 7,
            'I' => 8,
            'J' => 9,
            'K' => 10,
            'L' => 11,
            'M' => 12,
            'N' => 13,
            'O' => 14,
            'P' => 15,
            'Q' => 16,
            'R' => 17,
            'S' => 18,
            'T' => 19,
            'U' => 20,
            'V' => 21,
            'W' => 22,
            'X' => 23,
            'Y' => 24,
            'Z' => 25,
        ];

        $lastChar = [
            0  => 'A',
            1  => 'B',
            2  => 'C',
            3  => 'D',
            4  => 'E',
            5  => 'F',
            6  => 'G',
            7  => 'H',
            8  => 'I',
            9  => 'J',
            10 => 'K',
            11 => 'L',
            12 => 'M',
            13 => 'N',
            14 => 'O',
            15 => 'P',
            16 => 'Q',
            17 => 'R',
            18 => 'S',
            19 => 'T',
            20 => 'U',
            21 => 'V',
            22 => 'W',
            23 => 'X',
            24 => 'Y',
            25 => 'Z',
        ];

        $sum = 0;
        for ($i = 0; $i < 15; $i++) {
            $char = substr($code, $i, 1);
            $sum += ($i % 2) ? $even[$char] : $odd[$char];
        }

        return $lastChar[$sum % 26] == substr($code, 15, 1);
    }

    /**
     * @param string $code
     * @return bool
     *
     * @link http://www.icosaedro.it/cf-pi/
     */
    public static function partita_iva($code)
    {
        $code = trim((string) $code);

        if ($code === '') {
            return true;
        }

        // Lunghezza
        if (strlen($code) != 11) {
            return false;
        }

        // Solo numeri
        if (!preg_match("#^\\d+\$#", $code)) {
            return false;
        }

        // Somma dei numeri in posizione dispari
        $x = 0;
        for ($i = 0; $i <= 9; $i += 2) {
            $x += intval(substr($code, $i, 1));
        }

        // Somma dei numeri in posizione pari
        $y = 0;
        $z = 0;
        for ($i = 1; $i <= 9; $i += 2) {
            $p = intval(substr($code, $i, 1));
            $y += $p;

            // Se $y e' maggiore di 5, incremento $z
            if ($p >= 5) {
                $z++;
            }
        }

        // Radoppio la somma dei pari
        $y *= 2;

        // Sommo il tutto e calcolo il resto 10
        $mod = ($x + $y + $z) % 10;

        return (10 - $mod) % 10 == intval(substr($code, 10, 1));
    }

    /**
     * @param string $iban
     * @return bool
     */
    public static function iban($iban)
    {
        $iban = trim((string) $iban);

        if ($iban === '') {
            return true;
        }

        $iban = strtoupper($iban);

        return (bool) preg_match('#^IT(\d{2})([A-Z]{1})(\d{5})(\d{5})([0-9A-Z]{12})#', $iban);
    }

    /**
     * Questo metodo controlla la validità del codice SdI passato.
     *
     * Non ho trovato nessuna documentazione su come controllare il formato di questo codice, per tanto preparo solo
     * una funzione che restituisce sempre 'true'.
     *
     * @param string $sdi
     * @return bool
     */
    public static function sdi($sdi)
    {
        return true;
    }
}
