<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Utilities;

/**
 * Gestisce gli shortcode.
 */
final class Shortcode
{
    /**
     * Lista dei tag.
     *
     * @var array
     */
    protected $tags = [];

    /**
     * Aggiunge un tag alla lista.
     *
     * @param string   $tagName  Nome del tag.
     * @param callable $callback La callback da usare per il parsing dello shortcode.
     * @return $this
     */
    public function add($tagName, callable $callback)
    {
        $this->tags[$tagName] = $callback;

        return $this;
    }

    /**
     * Rimuove un tag.
     *
     * @param string $tagName Nome del tag da rimuovere.
     * @return $this
     */
    public function remove($tagName)
    {
        unset($this->tags[$tagName]);

        return $this;
    }

    /**
     * Rimuove tutti i tag.
     *
     * @return $this
     */
    public function removeAll()
    {
        $this->tags = [];

        return $this;
    }

    /**
     * Elabora un testo con gli shortcode.
     *
     * @param string $text       Testo da elaborare.
     * @param array  $attributes Attributi aggiuntvi da passare allo shortcode.
     * @return string Il testo elaborato.
     */
    public function parse(string $text, array $attributes = []): string
    {
        if ($this->tags === []) {
            return $text;
        }

        return preg_replace_callback('/' . $this->getRegex() . '/s', function ($match) use ($attributes) {
            return $this->parseTag($match, $attributes);
        }, $text);
    }

    /**
     * Rimuove gli shortcode da un testo.
     *
     * @param string $text Testo da elaborare.
     * @return string Il testo senza shortcode.
     */
    public function strip($text)
    {
        if ($this->tags === []) {
            return $text;
        }

        return preg_replace('/' . $this->getRegex() . '/s', '', $text);
    }

    /**
     * Metodo interno di callback usato dal metodo parse().
     *
     * @param array $match      Risultato dell’espressione regolare sul testo da elaborare.
     * @param array $attributes Attributi aggiuntivi.
     * @return string
     */
    private function parseTag(array $match, array $attributes = [])
    {
        $tagName = $match[3];
        $attributes = array_merge($attributes, $this->parseAttributes($match[4]));
        $content = $match[7] ?? '';

        return call_user_func($this->tags[$tagName], $attributes, $content, $tagName);
    }

    /**
     * Restituisce l'espressione regolare per il parsing del testo.
     *
     * Deve soddisfare questi codici:
     * <code>
     *   [tag /]
     *   [tag attr="value" /]
     *   [tag] ... [/tag]
     *   [tag attr="value"] ... [/tag]
     * </code>
     *
     * @return string L’espressione regolare per la ricerca degli shortcode.
     */
    private function getRegex()
    {
        $tagNames = array_keys($this->tags);
        $regex = implode('|', array_map('preg_quote', $tagNames));

        return '(<(p)>)?\[(' . $regex . ')([^\\]\\/]*(?:\\/(?!\\])[^\\]\\/]*)*?)(?:( \/\])|(\](.*?)\[\/\3\]))(<\/\2>)?';
    }

    /**
     * Converte una stringa di attributi in array.
     *
     * @param string $text Stringa di attributi da elaborare.
     * @return array L’elenco degli attributi.
     */
    private function parseAttributes($text)
    {
        $text = trim($text);

        if ($text === '') {
            return [];
        }

        preg_match_all('/(.*?)=[\"\'](.*?)[\"\']/', $text, $matches, PREG_SET_ORDER);

        $attributes = [];
        foreach ($matches as $match) {
            $attributes[trim($match[1])] = trim($match[2]);
        }

        return $attributes;
    }
}
