<?php

namespace Application\Core\Utilities;

use Application\Core\AppKernel;

class Pagination implements \Stringable
{
    protected $url_base;

    protected $total_pages;

    protected $total_records;

    protected $current_page;

    protected $query_string;

    protected $options = [
        'view_first'             => null, // null | always | never
        'view_prev'              => null, // null | always | never
        'view_next'              => null, // null | always | never
        'view_last'              => null, // null | always | never
        'view_page_of'           => false, // false | after | before
        'view_total_records'     => false, // true | false
        'view_total_records_pos' => 'before', // after | before
    ];

    public function __construct($url_base, $total_records, $current_page, $records_per_page = 10, array $query_string = [])
    {
        $this->url_base = $url_base;
        $this->current_page = (int) $current_page;
        $this->query_string = $query_string;
        $this->total_records = $total_records;
        $this->total_pages = ceil($total_records / $records_per_page);
    }

    public function setOptions(array $options)
    {
        foreach ($options as $name => $value) {
            $this->setOption($name, $value);
        }

        return $this;
    }

    public function setOption($name, $value)
    {
        $this->options[$name] = $value;
    }

    public function getOption($name, $default = null)
    {
        return array_key_exists($name, $this->options) ? $this->options[$name] : $default;
    }

    public function __toString(): string
    {
        return (string) $this->getOutput();
    }

    public function getCurrentPage()
    {
        return $this->current_page;
    }

    public function getTotalRecords()
    {
        return $this->total_records;
    }

    public function getTotalPages()
    {
        return $this->total_pages;
    }

    public function getOutput($pagination_start = 2, $pagination_middle = 1, $pagination_end = 2)
    {
        // Se ho solo una pagina, non ha senso creare la paginazione
        if ($this->total_pages <= 1) {
            return '';
        }

        // Controllo che la pagina corrente sia interna al range 1 / ultima pagina
        if ($this->current_page < 1) {
            $this->current_page = 1;
        }

        if ($this->current_page > $this->total_pages) {
            $this->current_page = $this->total_pages;
        }

        // Raccolgo in un array le pagine che devo visualizzare
        $pages = [];

        // Pagine all'inizio
        for ($i = 1; $i <= $pagination_start; $i++) {
            $pages[] = $i;
        }

        // Pagine alla fine
        for ($i = $this->total_pages - $pagination_end + 1; $i <= $this->total_pages; $i++) {
            $pages[] = $i;
        }

        // Pagine precedenti e successive a quella corrente
        for ($i = 1; $i <= $pagination_middle; $i++) {
            $pages[] = $this->current_page - $i;
            $pages[] = $this->current_page + $i;
        }

        // Pagina corrente
        $pages[] = $this->current_page;

        // Unifico e metto il tutto in ordine...
        $pages = array_unique($pages, SORT_NUMERIC);
        sort($pages, SORT_NUMERIC);

        // Recupero la lingua per comodità
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = AppKernel::instance()->getContainer()->get('translator');

        // Codice della paginazione
        $pagination = '';

        // Pagina X di Y
        if ($this->getOption('view_page_of')) {
            $page_of = '<div class="pages">' . $translator->trans(
                'Page %page% of %pages%',
                [
                    '%page%'  => $this->current_page,
                    '%pages%' => $this->total_pages,
                ]
            ) . '</div>';
        } else {
            $page_of = '';
        }

        // X record
        if ($this->getOption('view_total_records')) {
            $total_records = '<li class="total">' . sprintf($this->getOption('view_total_records', '%s'), $this->total_records) . '</li>';
        } else {
            $total_records = '';
        }

        if ($this->getOption('view_page_of') === 'before') {
            $pagination .= $page_of;
        }

        $pagination .= '<ul>';

        if ($this->getOption('view_total_records_pos') === 'before') {
            $pagination .= $total_records;
        }

        // Prima pagina
        match ($this->getOption('view_first', null)) {
            'always' => $pagination .= $this->current_page == 1 ? '<li class="first"><span>' . $translator->trans('« First') . '</span></li>' : '<li class="first"><a href="' . $this->page_url(1) . '">' . $translator->trans('« First') . '</a></li>',
            'never'  => $pagination .= '',
            default  => $pagination .= $this->current_page == 1 ? '' : '<li class="first"><a href="' . $this->page_url(1) . '">' . $translator->trans('« First') . '</a></li>',
        };

        // Pagina precedente
        match ($this->getOption('view_prev', null)) {
            'always' => $pagination .= $this->current_page == 1 ? '<li class="prev"><span>' . $translator->trans('‹ Previous') . '</span></li>' : '<li class="prev"><a rel="prev" href="' . $this->page_url($this->current_page - 1) . '">' . $translator->trans('‹ Previous') . '</a></li>',
            'never'  => $pagination .= '',
            default  => $pagination .= $this->current_page == 1 ? '' : '<li class="prev"><a rel="prev" href="' . $this->page_url($this->current_page - 1) . '">' . $translator->trans('‹ Previous') . '</a></li>',
        };

        // Pagine normali
        $i = 1;
        foreach ($pages as $page) {
            // Proseguo con la pagina successiva se questa non è compresa nel range 1 ~ $this->total_pages
            if ($page < 1 || $page > $this->total_pages) {
                continue;
            }

            // Inserisco dei puntini tra le pagine distanti
            if ($i != $page) {
                $pagination .= '<li class="dots"><span>…</span></li>';
                $i = $page;
            }

            $classes = [];
            if ($this->current_page == $page) {
                $classes[] = 'current';
                $pagination .= '<li class="' . implode(' ', $classes) . '"><span>' . $page . '</span></li>';
            } else {
                $pagination .= ($classes === [] ? '<li>' : '<li class="' . implode(' ', $classes) . '">') . '<a href="' . $this->page_url($page) . '">' . $page . '</a></li>';
            }

            $i++;
        }

        // Pagina successiva
        match ($this->getOption('view_next', null)) {
            'always' => $pagination .= $this->current_page == $this->total_pages ? '<li class="next"><span>' . $translator->trans('Next ›') . '</span></li>' : '<li class="next"><a rel="next" href="' . $this->page_url($this->current_page + 1) . '">' . $translator->trans('Next ›') . '</a></li>',
            'never'  => $pagination .= '',
            default  => $pagination .= $this->current_page == $this->total_pages ? '' : '<li class="next"><a rel="next" href="' . $this->page_url($this->current_page + 1) . '">' . $translator->trans('Next ›') . '</a></li>',
        };

        // Ultima pagina
        match ($this->getOption('view_last', null)) {
            'always' => $pagination .= $this->current_page == $this->total_pages ? '<li class="last"><span>' . $translator->trans('Last »') . '</span></li>' : '<li class="last"><a href="' . $this->page_url($this->total_pages) . '">' . $translator->trans('Last »') . '</a></li>',
            'never'  => $pagination .= '',
            default  => $pagination .= $this->current_page == $this->total_pages ? '' : '<li class="last"><a href="' . $this->page_url($this->total_pages) . '">' . $translator->trans('Last »') . '</a></li>',
        };

        if ($this->getOption('view_total_records_pos') === 'after') {
            $pagination .= $total_records;
        }

        $pagination .= '</ul>';

        if ($this->getOption('view_page_of') === 'after') {
            $pagination .= $page_of;
        }

        return $pagination;
    }

    private function page_url($page)
    {
        $url = $this->url_base;

        $query = '';
        if (!empty($this->query_string)) {
            $query = '?' . http_build_query($this->query_string);
        }

        $url .= ($page === 1 ? '' : "page/{$page}/") . $query;

        return url($url);
    }
}
