<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Utilities;

use Pongho\Core\ControllerHelper as Base;
use Pongho\Core\TemplateResponse;
use Pongho\Form\Form;
use Pongho\Http\JsonResponse;
use Pongho\Http\RedirectResponse;
use Pongho\Template\View;

class ControllerHelper extends Base
{
    /**
     * @var \Application\Core\Model\ModuleSite
     */
    protected $moduleSite;

    /**
     * @var \Pongho\Template\Theme
     */
    protected $theme;

    /**
     * @return string
     */
    public function getPath()
    {
        $path = trim((string) $this->container->getParameter('path'), '/');

        return $path === '' ? '/' : '/' . $path . '/';
    }

    /**
     * @return \Pongho\Template\HeadHelper
     */
    public function getThemeHeaderHelper()
    {
        return $this->container->get('template_head');
    }

    /**
     * @return \Pongho\Template\FooterHelper
     */
    public function getThemeFooterHelper()
    {
        return $this->container->get('template_footer');
    }

    /**
     * @param string $link
     * @return $this
     */
    public function addJavascript($link, array $attributes = [])
    {
        /** @var \Pongho\Template\FooterHelper $templateFooter */
        $templateFooter = $this->container->get('template_footer');

        $templateFooter->javascript->add($link, $attributes);

        return $this;
    }

    /**
     * @param string $code
     * @return $this
     */
    public function addJavascriptInline($code, array $attributes = [])
    {
        /** @var \Pongho\Template\FooterHelper $templateFooter */
        $templateFooter = $this->container->get('template_footer');

        $templateFooter->javascript->addInline($code, $attributes);

        return $this;
    }

    /**
     * @param string $link
     * @return $this
     */
    public function addCss($link, array $attributes = [])
    {
        /** @var \Pongho\Template\HeadHelper $templateHeader */
        $templateHeader = $this->container->get('template_head');

        $templateHeader->css->add($link, $attributes);

        return $this;
    }

    /**
     * Restituisce la vista.
     *
     * @return \Pongho\Template\Theme
     */
    public function getTheme()
    {
        if (null === $this->theme) {
            // Il controller potrebbe non essere ancora impostato, ad esempio quando si sta
            // usando il filtro "core.replace_handle". In ogni caso, non ricordo di aver mai
            // usato il subject negli eventi!
            $subject = $this->container->has('controller') ? $this->container->get('controller') : null;

            $this->theme = $this->filter(
                $subject,
                'core.filter_view',
                $this->container->get('theme_view')
            );

            $this->theme->assignVars([
                'module' => $this->getModuleSite(false),
            ]);
        }

        return $this->theme;
    }

    /**
     * @param string $templatePath
     * @return View
     */
    public function createView($templatePath)
    {
        $view = new View($templatePath);
        $view->assignVars($this->container->get('global_view_vars'));

        return $view;
    }

    /**
     * Restituisce una risposta HTTP con un messaggio.
     *
     * @param string $message
     * @param bool   $isError
     * @return \Pongho\Http\Response
     */
    public function displayMessage($message, $isError = false, array $parameters = [])
    {
        if ($this->container->get('request')->isAjax()) {
            return $this->displayJsonMessage($message, $isError, $parameters);
        }

        $parameters['message'] = $message;
        $parameters['error'] = $isError;

        $this->getTheme()
            ->setTemplate('message.php')
            ->assignVars($parameters);

        return new TemplateResponse($this->getTheme(), $this->getResponseHeaders());
    }

    /**
     * Restituisce una risposta JSON con un messaggio.
     *
     * @param string $message
     * @param bool   $isError
     * @return JsonResponse
     */
    public function displayJsonMessage($message, $isError = false, array $parameters = [])
    {
        $parameters['message'] = $message;
        $parameters['error'] = $isError;

        return new JsonResponse($parameters);
    }

    /**
     * Restituisce una risposta HTTP con un messaggio di errore.
     *
     * @param string $message
     * @return \Pongho\Http\Response
     */
    public function displayError($message, array $parameters = [])
    {
        return $this->displayMessage($message, true, $parameters);
    }

    /**
     * Restituisce una risposta JSON con un messaggio di errore.
     *
     * @param string $message
     * @return JsonResponse
     */
    public function displayJsonError($message, array $parameters = [])
    {
        return $this->displayJsonMessage($message, true, $parameters);
    }

    /**
     * Restituisce una risposta JSON con un elenco di errori.
     *
     * @return JsonResponse
     */
    public function displayJsonErrors(array $errors)
    {
        /** @var \Application\Core\Localization $lang */
        $lang = $this->container->get('language');

        return $this->displayJsonError($lang->get('error_title', count($errors)), ['errors' => $errors]);
    }

    /**
     * Restituisce una risposta JSON con gli errori di una form.
     *
     * @return JsonResponse
     */
    public function displayJsonFormErrors(Form $form)
    {
        $errors = [];
        foreach ($form->getErrors() as $key => $errorMessage) {
            if ($form->hasField($key)) {
                $errors[$form->getField($key)->getName()] = $errorMessage;
            } else {
                $errors[] = $errorMessage;
            }
        }

        return $this->displayJsonErrors($errors);
    }

    /**
     * Restituisce una risposta per un redirect all’url passato.
     *
     * Questo metodo tiene conto anche delle richieste AJAX.
     *
     * @param string $redirectUrl
     * @return \Pongho\Http\Response
     */
    public function redirectResponse($redirectUrl)
    {
        $redirectUrl = absolute_url($redirectUrl);

        if ($this->container->get('request')->isAjax()) {
            return new JsonResponse([
                'redirect' => $redirectUrl,
            ]);
        } else {
            return new RedirectResponse($redirectUrl);
        }
    }

    /**
     * Restituisce una risposta di redirect alla pagina di login.
     *
     * Specificando un valore per `$redirectUrl`, dopo il login l’utente verrà reindirizzato alla pagina indicata.
     *
     * @param string $redirectUrl
     * @return \Pongho\Http\Response
     */
    public function redirectToLogin($redirectUrl = null)
    {
        return $this->redirectResponse(
            $this->getLoginUrl($redirectUrl)
        );
    }

    /**
     * @return string
     */
    public function getLoginUrl($redirectUrl = null)
    {
        $loginUrl = rtrim($this->getLanguage()->path, '/') . '/user/login/';
        $loginUrl = $this->filter(null, 'user.login_url', $loginUrl);

        if ($redirectUrl) {
            $loginUrl .= '?' . http_build_query(['redirect' => $redirectUrl]);
        }

        return $loginUrl;
    }

    /**
     * Inizializza la lingua.
     *
     * @see \Application\Core\Entity\LanguageInterface::startLanguage()
     *
     * @return $this
     */
    public function startLanguage()
    {
        $this->getSite()->getDefaultLanguage()->startLanguage();
    }

    /**
     * Restituisce la localizzazione.
     *
     * @return \Application\Core\Localization
     */
    public function getLocalization()
    {
        return $this->container->get('language');
    }

    /**
     * @return \Application\Core\I18n\Translation\Translator
     */
    public function getTranslator()
    {
        return $this->container->get('translator');
    }

    /**
     * Restituisce il modello della lingua.
     *
     * @return \Application\Core\Model\LanguageSite
     */
    public function getLanguage()
    {
        return $this->container->get('language_site');
    }

    /**
     * Restituisce l'id della lingua
     *
     * @return int
     */
    public function getLanguageId()
    {
        return $this->getLanguage()->getLanguageId();
    }

    /**
     * Resituisce il modello del sito corrente.
     *
     * @return \Application\Core\Model\Site
     */
    public function getSite()
    {
        return $this->container->get('site');
    }

    /**
     * Restituisce l’ID del sito corrente.
     *
     * @return int
     */
    public function getSiteId()
    {
        return $this->getSite()->getId();
    }

    /**
     * Restituisce l’utente.
     *
     * @return \Application\Core\User
     */
    public function getUser()
    {
        return $this->container->get('current_user');
    }

    /**
     * Restituisce l’ID dell'utente corrente.
     *
     * @return int
     */
    public function getUserId()
    {
        return $this->getUser()->getAccount()->getId();
    }

    /**
     * Verifica se l'utente ha un permesso specifico
     *
     * @param string $key
     * @return bool
     */
    public function userHasPermit($key)
    {
        return $this->getUser()->hasPermit($key);
    }

    /**
     * Restituisce il modulo legato al sito.
     *
     * @param bool $thrown_exception
     * @return \Application\Core\Model\ModuleSite
     */
    public function getModuleSite($thrown_exception = true)
    {
        if (null === $this->moduleSite) {
            $this->moduleSite = $this->container->getParameter('site_module', null, $thrown_exception);
        }

        return $this->moduleSite;
    }

    /**
     * @return \Pongho\Template\HeadHelper
     *
     * @deprecated
     */
    public function getHead()
    {
        trigger_error('Method ' . __METHOD__ . ' is deprecated, use getThemeHeaderHelper() instead', E_USER_DEPRECATED);

        return $this->getThemeHeaderHelper();
    }

    /**
     * Restituisce la vista.
     *
     * @return \Pongho\Template\Theme
     *
     * @deprecated
     */
    public function getView()
    {
        trigger_error('Method ' . __METHOD__ . ' is deprecated, use getTheme() instead', E_USER_DEPRECATED);

        return $this->getTheme();
    }
}
