<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Utilities;

use Application\Core\Entity\UserDetailsInterface;
use Application\Core\I18n\Translation\Translator;
use Pongho\Utilities\DateTime;

class AccountInfoRenderer
{
    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var UserDetailsInterface
     */
    protected $user_details;

    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
    }

    public function setAccount(UserDetailsInterface $userDetails)
    {
        $this->user_details = $userDetails;
    }

    /**
     * @return string
     */
    public function render()
    {
        if ($this->user_details === null) {
            throw new \LogicException('You must set the Account to render its details');
        }

        return <<<HTML
<div id="account-info" class="panel">
    {$this->getAvatarCode()}
    <div id="account-info-content">
        {$this->getTitleCode()}
        {$this->getContentCode()}
    </div>
</div>
HTML;
    }

    /**
     * @return string
     */
    protected function getAvatarCode()
    {
        $avatarSrc = src(
            $this->user_details->getAvatar(),
            'adminpreview',
            pongho_url('/Application/Core/Resources/images/avatar-ghost.png')
        );

        return '<img src="' . $avatarSrc . '">';
    }

    /**
     * @return string
     */
    protected function getTitleCode()
    {
        $name = $this->user_details->name() ?: $this->translator->trans('New account');

        return '<h3>' . $name . '</h3>';
    }

    /**
     * @return string
     */
    protected function getContentCode()
    {
        return <<<HTML
<p>{$this->translator->trans('Subscription: %date%', ['%date%' => $this->getSubscribeDate()])}</p>
<p>{$this->translator->trans('Last login: %date%', ['%date%' => $this->getLastLoginDate()])}</p>
HTML;
    }

    /**
     * @return string
     */
    protected function getLastLoginDate()
    {
        $lastLogin = $this->user_details->getAccount()->getLastLoginDate();

        return $lastLogin instanceof DateTime ? $lastLogin->format('d-m-Y') : $this->translator->trans('never');
    }

    /**
     * @return string
     */
    protected function getSubscribeDate()
    {
        $subscribeDate = $this->user_details->getAccount()->getSubscribeDate();

        if ($subscribeDate instanceof DateTime) {
            return $subscribeDate->format('d-m-Y');
        }

        return (new DateTime())->format('d-m-Y');
    }
}
