<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core;

use Application\Core\Entity\AccountInterface;
use Application\Core\Model\Account;
use Pongho\Http\Session;
use Pongho\Utilities\DateTime;

/**
 * Rappresenta l’utente corrente.
 */
class User
{
    /**
     * Modello Account.
     *
     * @var AccountInterface
     */
    private $data;

    /**
     * @var Session
     *
     * @refactoring Si potrebbe utilizzare la nuova SessionInterface
     */
    protected $session;

    public function __construct(Session $session)
    {
        $this->session = $session;

        $this->setData();
    }

    /**
     * Cerca l’utente corrente e crea l’istanza con i dati in database.
     */
    private function setData()
    {
        $userId = (int) $this->getSession()->get('user_id');

        if ($userId <= 0) {
            $userId = Account::ANONYMOUS;
        }

        $this->data = Account::find($userId);

        if (null === $this->data || $this->data->isActive() === false) {
            $this->data = Account::find(Account::ANONYMOUS);
        }

        $this->getSession()->set('user_id', $this->data->getId());
    }

    /**
     * Restituisce il modello dell’account legato all’utente corrente.
     *
     * @return AccountInterface
     */
    public function getAccount()
    {
        return $this->data;
    }

    /**
     * Restituisce un attributo del modello account dell’utente corrente.
     *
     * @param string $name
     * @return mixed
     */
    public function __get($name)
    {
        try {
            return $this->data->$name;
        } catch (\Exception) {
            return $name;
        }
    }

    /**
     * Esegue il login.
     *
     * @param string $usernameOrEmail
     * @param string $password
     * @param bool   $autologin
     * @param array  $errors
     * @return boolean
     */
    public function login(&$usernameOrEmail, &$password, $autologin, &$errors)
    {
        $lang = $this->getLanguage();

        // Campo nome utente o email obbligatorio
        if (empty($usernameOrEmail)) {
            $errors['username_or_email'] = $lang->get('login_required_username_or_email');
            $usernameOrEmail = '';

            return false;
        }

        // Campo password obbligatorio
        if (empty($password)) {
            $errors['password'] = $lang->get('login_required_password');
            $password = '';

            return false;
        }

        if (is_email($usernameOrEmail)) {
            // Ricavo l'account dall'indirizzo e-mail
            $userdata = Account::findByEmail($usernameOrEmail);

            // Nessun account trovato
            if ($userdata === null) {
                $errors['username_or_email'] = $lang->get('login_email_not_found');
                $usernameOrEmail = '';

                return false;
            }
        } else {
            // Ricavo l'account dal nome utente
            $userdata = Account::findByUsername($usernameOrEmail);

            // Nessun account trovato
            if ($userdata === null) {
                $errors['username_or_email'] = $lang->get('login_username_not_found');
                $usernameOrEmail = '';

                return false;
            }
        }

        // Confronto la password dell'account con quella passata
        if ($userdata->getPassword() != $password) {
            $errors['password'] = $lang->get('login_password_not_valid');
            $password = '';

            return false;
        }

        // Utente attivo?
        if (!$userdata->isActive()) {
            $errors['active'] = $lang->get('login_user_not_active');
            $usernameOrEmail = '';
            $password = '';

            return false;
        }

        // Aggiorna la data di ultimo login
        $userdata->last_login_at = new DateTime();
        $userdata->save();

        // Inizializzo l'utente
        $this->data = $userdata;
        $this->getSession()->set('user_id', $this->data->getId());

        return true;
    }

    /**
     * Forza il login all’account passato per gestire l’accesso automatico dopo alcune operazioni particolari.
     *
     * Ovviamente, **DA USARE CON CAUTELA**!
     *
     * @return $this
     */
    public function forceLogin(AccountInterface $account)
    {
        $this->data = $account;
        $this->getSession()->set('user_id', $account->getId());

        return $this;
    }

    /**
     * Esegue il logout dell’utente.
     *
     * @param bool $continue Indica se dopo il logout lo script continua, oppure esce.
     */
    public function logout($continue = false)
    {
        $this->getSession()->destroy();

        if ($continue) {
            $this->data = Account::find(Account::ANONYMOUS);
        }
    }

    /**
     * Modifica i dati dell’utente corrente.
     *
     * @return bool
     */
    public function edit(array $data = [])
    {
        $this->data->updateAttributes($data);

        if ($this->data->save()) {
            return true;
        }

        return $this->data->errors;
    }

    /**
     * Restituisce il nome dell’utente.
     *
     * @return string
     */
    public function name()
    {
        return $this->data->name();
    }

    /**
     * Restituisce true se l’utente è loggato.
     *
     * @return bool
     */
    public function isLogged()
    {
        return ($this->data->getId() !== Account::ANONYMOUS);
    }

    /**
     * Restituisce true se l’utente è amministratore.
     *
     * @return bool
     */
    public function isAdmin()
    {
        return $this->data->isAdmin();
    }

    /**
     * Restituisce true se l’utente è founder.
     *
     * @return bool
     */
    public function isFounder()
    {
        return $this->data->isFounder();
    }

    /**
     * Restituisce true se l’utente ha il permesso indicato.
     *
     * @param string $key
     * @return bool
     */
    public function hasPermit($key)
    {
        return $this->data->hasPermit($key);
    }

    /**
     * Verifica se l’utente appartiene ad un gruppo.
     *
     * È possibile specificare anche una lista di gruppi passando come argomento un array.
     * Il gruppo può essere sia istanza di Group che l'ID del gruppo.
     *
     * @param mixed $groups
     * @return bool
     */
    public function inGroup($groups)
    {
        return $this->data->inGroup($groups);
    }

    /**
     * Restituisce il gestore della sessione.
     *
     * @return \Pongho\Http\SessionInterface
     */
    public function getSession()
    {
        return $this->session;
    }

    /**
     * Restituisce il gestore della localizzazione.
     *
     * @return Localization
     */
    protected function getLanguage()
    {
        // Problema di dipendenze circolari, "language" dipende da "user", non posso passargli il servizio
        return AppKernel::instance()->getContainer()->get('language');
    }
}
