<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Tests\Utilities\Link;

use Application\Core\Entity\LinkTypeDetails;
use Application\Core\LinkType\Handler\ModelHandler;
use Application\Core\LinkType\LinkSelectOption;
use Application\Core\Tests\Utilities\Link\Utilities\Mocks;
use Application\Core\Utilities\Html;
use Symfony\Component\DomCrawler\Crawler;

class ModelHandlerTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var Html
     */
    protected $html;

    /**
     * @var Mocks
     */
    protected $mocks;

    protected function setUp(): void
    {
        $this->html = new Html();
        $this->mocks = new Mocks($this);
    }

    public function testInitialization(): void
    {
        $expected_name = 'Name';
        $expected_label = 'Label';
        $options = [];

        $handler = $this->getHandler($expected_name, $expected_label, $options);

        $this->assertEquals($expected_name, $handler->getName());
        $this->assertEquals($expected_label, $handler->getLabel());
    }

    public function testGetDetailsFromPost(): void
    {
        $handler = $this->getHandler('model', '', []);

        $expected_language_id = 123;
        $expected_model_id = 321;

        $post = [
            'language_id' => $expected_language_id,
            'model'       => $expected_model_id,
        ];

        $details = $handler->getDetails($post);

        $this->assertInstanceOf(\Application\Core\Entity\LinkTypeDetails::class, $details);
        $this->assertEquals($expected_language_id, $details->getLanguageId());
        $this->assertEquals($expected_model_id, $details->getModelId());
    }

    public function testGetDetailsThrowsExceptionForMissingLanguageInPost(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $handler = $this->getHandler('', '', []);

        $post = [
            'model_id' => 123,
        ];

        $handler->getDetails($post);
    }

    public function testGetDetailsNormalizeTheModelToNull(): void
    {
        $handler = $this->getHandler('', '', []);

        $post = [
            'language_id' => 123,
        ];

        $this->assertNull($handler->getDetails($post)->getModelId());
    }

    public function testRenderSelectWithOptions(): void
    {
        /** @var LinkSelectOption[] $options */
        $options = $this->getLinkSelectOptions();
        $handler = $this->getHandler('test', 'Test models', $options);

        $render = $handler->render('form[fieldset][link_type]', null);

        $this->assertRenderContainsSelectWithDefinedOptions($render, $options);

        $crawler = new Crawler($render);

        /** @var \DOMElement $node */
        foreach ($crawler->filter('option') as $node) {
            if ($node->hasAttribute('value')) {
                if (!array_key_exists($node->getAttribute('value'), $options)) {
                    $this->fail('The option "' . $node->getAttribute('value') . '" could not be found in the defined array, check the test');
                }

                $label = $options[$node->getAttribute('value')]->getLabel();
                $this->assertEquals($label, $node->textContent);
            }
        }
    }

    public function testGetDefinedLink(): void
    {
        /** @var LinkSelectOption[] $options */
        $options = [
            new LinkSelectOption(1, 'A model', '/path/to/model/a'),
            new LinkSelectOption(2, 'B model', '/path/to/model/b'),
            new LinkSelectOption(3, 'C model', '/path/to/model/c'),
            new LinkSelectOption(4, 'D model', '/path/to/model/d'),
            new LinkSelectOption(5, 'E model', '/path/to/model/e'),
        ];

        $selected_id = 4;
        $expected_url = '/path/to/model/d';

        $handler_name = 'test';
        $handler = $this->getHandler($handler_name, '', $options);

        $post = [
            $handler_name => $selected_id,
        ];

        $this->assertEquals($expected_url, $handler->getUrl($post));
    }

    public function testGetValue(): void
    {
        $expected_value = 123;

        $handler_name = 'test';
        $handler = $this->getHandler($handler_name, '', []);

        $link_type = $this->mocks->getLinkTypeMock();
        $link_type->expects($this->any())
            ->method('getLinkDetails')
            ->willReturn(new LinkTypeDetails(null, $expected_value));
        $link_type->expects($this->any())->method('getIdentifier')->willReturn($handler_name);

        $this->assertInstanceOf(\Application\Core\Entity\LinkTypeInterface::class, $link_type);

        $this->assertEquals($expected_value, $handler->getValue($link_type));
    }

    /**
     * @param string $key
     * @param string $label
     * @param LinkSelectOption[] $models
     * @return ModelHandler
     */
    protected function getHandler($key, $label, array $models = [])
    {
        return new ModelHandler($key, $label, $models, $this->html);
    }

    /**
     * @return LinkSelectOption[]
     */
    protected function getLinkSelectOptions()
    {
        return [
            1 => new LinkSelectOption(1, 'A model', '/path/to/model/a'),
            2 => new LinkSelectOption(2, 'B model', '/path/to/model/b'),
        ];
    }

    /**
     * @param string             $render
     * @param LinkSelectOption[] $options
     */
    protected function assertRenderContainsSelectWithDefinedOptions($render, array $options)
    {
        $crawler = new Crawler($render);

        $this->assertEquals(1, $crawler->filter('select')->count());
        $this->assertEquals('form[fieldset][link_type]', $crawler->filter('select')->attr('name'));
        $this->assertEquals(count($options) + 1 /* empty placeholder */, $crawler->filter('option')->count());
    }
}
