<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Tests\Utilities\AssociativeArrayObject;

use Application\Core\Utilities\AssociativeArrayObject;

class AssociativeArrayObjectTest extends \PHPUnit\Framework\TestCase
{
    /** @var AssociativeArrayObject */
    protected $obj_assoc;

    /** @var array */
    protected $ary_assoc;

    /** @var AssociativeArrayObject */
    protected $obj_positional;

    /** @var array */
    protected $ary_positional;

    protected function setUp(): void
    {
        $this->ary_assoc = [
            'a' => 'A',
            'b' => 'B',
            'c' => 'C',
        ];

        $this->obj_assoc = new AssociativeArrayObject($this->ary_assoc);

        $this->ary_positional = [
            'A',
            'B',
            'C',
        ];

        $this->obj_positional = new AssociativeArrayObject($this->ary_positional);
    }

    public function testArray(): void
    {
        $this->assertEquals($this->ary_assoc, $this->obj_assoc->getArrayCopy());
        $this->assertEquals($this->ary_positional, $this->obj_positional->getArrayCopy());
    }

    public function testFirstPositional(): void
    {
        $this->obj_positional->appendFirst('X');

        $this->assertEquals(
            [
                'X',
                'A',
                'B',
                'C',
            ],
            $this->obj_positional->getArrayCopy()
        );

        $this->obj_positional->appendFirst('Y');

        $this->assertEquals(
            [
                'Y',
                'X',
                'A',
                'B',
                'C',
            ],
            $this->obj_positional->getArrayCopy()
        );
    }

    public function testFirstMixed(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendFirst('X');
    }

    public function testFirstWithKey(): void
    {
        $this->obj_assoc->appendFirst('X', 'x');

        $this->assertEquals(
            [
                'x' => 'X',
                'a' => 'A',
                'b' => 'B',
                'c' => 'C',
            ],
            $this->obj_assoc->getArrayCopy()
        );
    }

    public function testLastPositional(): void
    {
        $this->obj_positional->appendLast('X');

        $this->assertEquals(
            [
                'A',
                'B',
                'C',
                'X',
            ],
            $this->obj_positional->getArrayCopy()
        );

        $this->obj_positional->appendLast('Y');

        $this->assertEquals(
            [
                'A',
                'B',
                'C',
                'X',
                'Y',
            ],
            $this->obj_positional->getArrayCopy()
        );
    }

    public function testLastMixed(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendLast('X');
    }

    public function testLastWithKey(): void
    {
        $this->obj_assoc->appendLast('X', 'x');

        $this->assertEquals(
            [
                'a' => 'A',
                'b' => 'B',
                'c' => 'C',
                'x' => 'X',
            ],
            $this->obj_assoc->getArrayCopy()
        );
    }

    public function testAfterPositional(): void
    {
        $this->obj_positional->appendAfter(1, 'X');

        $this->assertEquals(
            [
                'A',
                'B',
                'X',
                'C',
            ],
            $this->obj_positional->getArrayCopy()
        );

        $this->obj_positional->appendAfter(1, 'Y');

        $this->assertEquals(
            [
                'A',
                'B',
                'Y',
                'X',
                'C',
            ],
            $this->obj_positional->getArrayCopy()
        );
    }

    public function testAfterMixed(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendAfter('b', 'X');
    }

    public function testAfterWithKey(): void
    {
        $this->obj_assoc->appendAfter('b', 'X', 'x');

        $this->assertEquals(
            [
                'a' => 'A',
                'b' => 'B',
                'x' => 'X',
                'c' => 'C',
            ],
            $this->obj_assoc->getArrayCopy()
        );
    }

    public function testBeforePositional(): void
    {
        $this->obj_positional->appendBefore(1, 'X');

        $this->assertEquals(
            [
                'A',
                'X',
                'B',
                'C',
            ],
            $this->obj_positional->getArrayCopy()
        );

        $this->obj_positional->appendBefore(1, 'Y');

        $this->assertEquals(
            [
                'A',
                'Y',
                'X',
                'B',
                'C',
            ],
            $this->obj_positional->getArrayCopy()
        );
    }

    public function testBeforeMixed(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendBefore('b', 'X');
    }

    public function testBeforeWithKey(): void
    {
        $this->obj_assoc->appendBefore('b', 'X', 'x');

        $this->assertEquals(
            [
                'a' => 'A',
                'x' => 'X',
                'b' => 'B',
                'c' => 'C',
            ],
            $this->obj_assoc->getArrayCopy()
        );
    }

    public function testAppendAfterKeyExists(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendAfter('a', 'X', 'b');
    }

    public function testAppendBeforeKeyExists(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendBefore('a', 'X', 'b');
    }

    public function testAppendFirstKeyExists(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendFirst('X', 'b');
    }

    public function testAppendLastKeyExists(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->obj_assoc->appendLast('X', 'b');
    }
}
