<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Tests\Template;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\SiteInterface;
use Application\Core\Template\TemplateHelperBuilder;
use PHPUnit\Framework\MockObject\MockObject;
use Pongho\Http\Request;
use Symfony\Component\DomCrawler\Crawler;

class TemplateHelperBuilderTest extends \PHPUnit\Framework\TestCase
{
    protected MockObject $site;

    protected MockObject $language;

    /**
     * @var TemplateHelperBuilder
     */
    protected $builder;

    protected function setUp(): void
    {
        $this->site = $this->createMock(SiteInterface::class);
        $this->language = $this->createMock(LanguageInterface::class);

        $this->site->method('getThemesPath')->willReturnCallback(function ($path) {
            return $path;
        });

        $this->builder = new TemplateHelperBuilder(Request::create('/'), $this->site, $this->language);
    }

    public function testBuildHeader(): void
    {
        $header = $this->builder->buildHeader();
        $this->assertInstanceOf(\Pongho\Template\HeadHelper::class, $header);
    }

    public function testBuildFooter(): void
    {
        $footer = $this->builder->buildFooter();
        $this->assertInstanceOf(\Pongho\Template\FooterHelper::class, $footer);
    }

    public function testBackwardCompatibility(): void
    {
        $header = $this->builder->buildHeader();
        $footer = $this->builder->buildFooter();

        $this->assertSame($header->js, $footer->javascript);
    }

    public function testBasicHeaderTags(): void
    {
        $this->prepareMocks(true);

        $crawler = new Crawler($this->builder->buildHeader()->output());

        // X-UA-Compatible
        $tag = $crawler->filter('meta[http-equiv="X-UA-Compatible"]');
        $this->assertCount(1, $tag);
        $this->assertEquals('IE=edge', $tag->attr('content'));

        // viewport
        $tag = $crawler->filter('meta[name="viewport"]');
        $this->assertCount(1, $tag);
        $this->assertEquals('width=device-width, initial-scale=1', $tag->attr('content'));

        // copyright
        $tag = $crawler->filter('meta[name="copyright"]');
        $this->assertCount(1, $tag);
        $this->assertEquals('(c) Copyright 2008-2015 - All rights reserved', $tag->attr('content'));

        // humans.txt
        $tag = $crawler->filter('link[rel="author"]');
        $this->assertCount(1, $tag);
        $this->assertEquals('text/plain', $tag->attr('type'));
        $this->assertEquals('http://www.example.com/humans.txt', $tag->attr('href'));

        // sitemap.xml
        $tag = $crawler->filter('link[rel="sitemap"]');
        $this->assertCount(1, $tag);
        $this->assertEquals('application/xml', $tag->attr('type'));
        $this->assertEquals('http://www.example.com/sitemap.xml', $tag->attr('href'));
        $this->assertEquals('Sitemap', $tag->attr('title'));
    }

    public function testOpenGraph(): void
    {
        $this->prepareMocks(true);

        // in Pongho
        $header = $this->builder->buildHeader();

        // nel template
        $header->setTitle('A.C.M.E. Page Title');
        $header->metatags->add('description', 'Lorem ipsum dolor sit amet');

        $output = $header->output();

        // Test
        $crawler = new Crawler($output);

        // og:title
        $og_title = $crawler->filter('meta[property="og:title"]');
        $this->assertCount(1, $og_title);
        $this->assertEquals('A.C.M.E. Page Title', $og_title->attr('content'));

        // og:type
        $og_type = $crawler->filter('meta[property="og:type"]');
        $this->assertCount(1, $og_type);
        $this->assertEquals('website', $og_type->attr('content'));

        // og:site_name
        $og_site_name = $crawler->filter('meta[property="og:site_name"]');
        $this->assertCount(1, $og_site_name);
        $this->assertEquals('A.C.M.E. Web Site', $og_site_name->attr('content'));

        // og:url
        $og_url = $crawler->filter('meta[property="og:url"]');
        $this->assertCount(1, $og_url);
        $this->assertEquals('http://www.example.com/', $og_url->attr('content'));

        // og:locale
        $og_locale = $crawler->filter('meta[property="og:locale"]');
        $this->assertCount(1, $og_locale);
        $this->assertEquals('it_IT', $og_locale->attr('content'));

        // og:description
        $og_description = $crawler->filter('meta[property="og:description"]');
        $this->assertCount(1, $og_description);
        $this->assertEquals('Lorem ipsum dolor sit amet', $og_description->attr('content'));
    }

    public function testFacebookEnabled(): void
    {
        $this->prepareMocks(true);

        $crawler = new Crawler($this->builder->buildHeader()->output());

        // fb:admins
        $fb_admins = $crawler->filter('meta[property="fb:admins"]');
        $this->assertCount(1, $fb_admins);
        $this->assertEquals('123', $fb_admins->attr('content'));

        // fb:app_id
        $fb_app_id = $crawler->filter('meta[property="fb:app_id"]');
        $this->assertCount(1, $fb_app_id);
        $this->assertEquals('456', $fb_app_id->attr('content'));
    }

    public function testFacebookDisabled(): void
    {
        $this->prepareMocks(false);

        $crawler = new Crawler($this->builder->buildHeader()->output());

        $this->assertCount(0, $crawler->filter('meta[property="fb:admins"]'));
        $this->assertCount(0, $crawler->filter('meta[property="fb:app_id"]'));
    }

    protected function prepareMocks($fb_enabled = true)
    {
        $options = [
            'fb'                => $fb_enabled,
            'fb_admin_id'       => '123',
            'fb_appid'          => '456',
            'fb_og_type'        => 'website',

            'site_name'         => 'A.C.M.E. Web Site',

            'company_copyright' => '(c) Copyright 2008-2015 - All rights reserved',
        ];

        $this->site->expects($this->any())
            ->method('getOption')
            ->willReturnCallback(
                function ($option) use ($options) {
                    return $options[$option] ?? '';
                }
            );

        $this->site->expects($this->any())
            ->method('getPermalink')
            ->willReturn('http://www.example.com/');

        $this->language->expects($this->any())
            ->method('getCulture')
            ->willReturn('it_IT');
    }
}
