<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

/**
 * Serve a {@see UserActivationLinkBuilder::build()}.
 */

namespace Application\Core\Utilities {
    if (!function_exists('Application\\Core\\Utilities\\absolute_url')) {
        function absolute_url(string $url): string
        {
            return $url;
        }
    }
}

namespace Application\Core\Tests\Notification {
    use Application\Core\Mailer\Helper;
    use Application\Core\Notification\Exception\SendException;
    use Application\Core\Notification\NotificationSender;
    use Application\Core\Notification\UserSubscribeEmailNotification;
    use Symfony\Component\Mailer\Exception\TransportExceptionInterface;

    class UserSubscribeNotificationTest extends AbstractNotificationUseCase
    {
        public function testNotificationEmailVars(): void
        {
            $site = $this->createSiteMock();
            $account = $this->createAccountMock();
            $translator = $this->createTranslatorMock();
            $generator = $this->createNotificationTemplateGeneratorMock();

            $notification = new UserSubscribeEmailNotification($site, $translator, $generator);
            $notification->setAccount($account);
            $notification->setPassword('foobar123');
            $notification->setAccountActivationLink('https://example.com/user/activation/');
            $notification->setSettingsLink('https://example.com/user/settings/');

            $vars = $notification->getEmailVars();

            // Controllo l'esistenza di solo alcuni valori, non mi interessano tutti quelli che va a inserire il
            // template perché potrebbero essere un'infinità, ma mi interessa che ci siano quelli che imposto nel test
            $expected_vars = [
                'USER_USERNAME'        => 'Pippo',
                'USER_EMAIL'           => 'pippo@email.com',
                'USER_PASSWORD'        => 'foobar123',
                'USER_ACTIVATION_LINK' => 'https://example.com/user/activation/',
                'USER_SETTINGS_LINK'   => 'https://example.com/user/settings/',
                'PONGHO_COMPANY_LOGO'  => 'https://example.com/theme/images/logo-email.png',
                'PONGHO_COMPANY_NAME'  => 'ACME',
                'PONGHO_COMPANY_EMAIL' => 'admin@website.com',
            ];

            foreach ($expected_vars as $key => $var) {
                $this->assertArrayHasKey($key, $vars);
                $this->assertEquals($var, $vars[$key]);
            }
        }

        public function testSendNotification(): void
        {
            $site = $this->createSiteMock();
            $account = $this->createAccountMock();
            $mailer = $this->createMailerMock();
            $logger = $this->createLoggerMock();
            $localization = $this->createLocalizationMock();
            $translator = $this->createTranslatorMock();
            $generator = $this->createNotificationTemplateGeneratorMock();

            $helper = new Helper($site, $localization);

            $notification = new UserSubscribeEmailNotification($site, $translator, $generator);
            $notification->setAccount($account);
            $notification->setPassword('foobar123');
            $notification->setAccountActivationLink('https://example.com/user/activation/');
            $notification->setSettingsLink('https://example.com/user/settings/');

            $template = $this->createNotificationTemplateMock(
                $notification->getName(),
                '%USER_USERNAME% <%USER_EMAIL%>',
                '%PONGHO_COMPANY_NAME% <%PONGHO_COMPANY_EMAIL%>',
                'A new user subscribed!',
                'User details: %USER_USERNAME% %USER_EMAIL%',
                '%PONGHO_COMPANY_NAME% <%PONGHO_COMPANY_EMAIL%>',
                '%USER_USERNAME% <%USER_EMAIL%>',
                'Welcome!',
                'Your details: %USER_USERNAME% %USER_EMAIL% %USER_PASSWORD%'
            );

            $templateManager = $this->createNotificationTemplateManagerMock($notification, $template);

            // Condizioni
            $mailer->expects($this->exactly(2))->method('send');
            $logger->expects($this->exactly(2))->method('info');

            // Esecuzione
            $sender = new NotificationSender($site, $mailer, $helper, $templateManager, $logger);
            $sender->send($notification);
        }

        public function testFailSendNotification(): void
        {
            $exception = new class('Simulated server email send error') extends \Exception implements TransportExceptionInterface {
                public function getDebug(): string
                {
                    return '';
                }

                public function appendDebug(string $debug): void
                {
                }
            };

            $this->expectException(SendException::class);
            $this->expectExceptionMessage('Simulated server email send error');

            $site = $this->createSiteMock();
            $account = $this->createAccountMock();
            $mailer = $this->createMailerMock();
            $logger = $this->createLoggerMock();
            $localization = $this->createLocalizationMock();
            $translator = $this->createTranslatorMock();
            $generator = $this->createNotificationTemplateGeneratorMock();

            $helper = new Helper($site, $localization);

            $notification = new UserSubscribeEmailNotification($site, $translator, $generator);
            $notification->setAccount($account);
            $notification->setPassword('foobar123');
            $notification->setAccountActivationLink('https://example.com/user/activation/');
            $notification->setSettingsLink('https://example.com/user/settings/');

            $template = $this->createNotificationTemplateMock(
                $notification->getName(),
                '%USER_USERNAME% <%USER_EMAIL%>',
                '%PONGHO_COMPANY_NAME% <%PONGHO_COMPANY_EMAIL%>',
                'A new user subscribed!',
                'User details: %USER_USERNAME% %USER_EMAIL%',
                '%PONGHO_COMPANY_NAME% <%PONGHO_COMPANY_EMAIL%>',
                '%USER_USERNAME% <%USER_EMAIL%>',
                'Welcome!',
                'Your details: %USER_USERNAME% %USER_EMAIL% %USER_PASSWORD%'
            );

            $templateManager = $this->createNotificationTemplateManagerMock($notification, $template);

            // Condizioni
            $mailer->expects($this->once())->method('send')->willThrowException($exception);
            $logger->expects($this->once())->method('error');

            // Esecuzione
            $sender = new NotificationSender($site, $mailer, $helper, $templateManager, $logger);
            $sender->send($notification);
        }
    }
}
