<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

declare(strict_types=1);

namespace Application\Core\Tests\Notification;

use Application\Core\Entity\AccountInterface;
use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\RoleInterface;
use Application\Core\Entity\SiteInterface;
use Application\Core\Entity\UserDetailsInterface;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Model\Manager\NotificationTemplateManagerInterface;
use Application\Core\Model\NotificationTemplate;
use Application\Core\Notification\NotificationInterface;
use Application\Core\Notification\NotificationTemplateGeneratorInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Pongho\Core\LocalizationInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Mailer\MailerInterface;

abstract class AbstractNotificationUseCase extends TestCase
{
    /**
     * @return AccountInterface|MockObject
     */
    protected function createAccountMock()
    {
        $account = $this->createMock(AccountInterface::class);
        $userDetails = $this->createMock(UserDetailsInterface::class);
        $role = $this->createMock(RoleInterface::class);

        $account->method('getUserDetails')->willReturn($userDetails);
        $account->method('getRole')->willReturn($role);

        $account->method('getUsername')->willReturn('Pippo');
        $account->method('getEmail')->willReturn('pippo@email.com');
        $account->method('getLanguage')->willReturn($this->createLanguageMock());

        return $account;
    }

    /**
     * @return LanguageInterface|MockObject
     */
    protected function createLanguageMock()
    {
        $language = $this->createMock(LanguageInterface::class);
        $language->method('getId')->willReturn(1);

        return $language;
    }

    /**
     * @return LocalizationInterface|MockObject
     */
    protected function createLocalizationMock()
    {
        $mock = $this->createMock(LocalizationInterface::class);

        $mock->method('get')->willReturnCallback(function ($string) {
            return $string;
        });

        return $mock;
    }

    /**
     * @return LoggerInterface|MockObject
     */
    protected function createLoggerMock()
    {
        return $this->createMock(LoggerInterface::class);
    }

    /**
     * @return MailerInterface|MockObject
     */
    protected function createMailerMock()
    {
        return $this->createMock(MailerInterface::class);
    }

    /**
     * @return NotificationTemplate|MockObject
     */
    protected function createNotificationTemplateMock(
        string $notificationName,
        string $fromAdmin = '',
        string $toAdmin = '',
        string $subjectAdmin = '',
        string $contentAdmin = '',
        string $fromUser = '',
        string $toUser = '',
        string $subjectUser = '',
        string $contentUser = ''
    ) {
        $template = $this->createMock(NotificationTemplate::class);

        $template->method('getNotificationName')->willReturn($notificationName);
        $template->method('getSenderToAdmin')->willReturn($fromAdmin);
        $template->method('getRecipientToAdmin')->willReturn($toAdmin);
        $template->method('getReplyToAdmin')->willReturn(''); // todo
        $template->method('getCcToAdmin')->willReturn(''); // todo
        $template->method('getBccToAdmin')->willReturn(''); // todo
        $template->method('getSubjectToAdmin')->willReturn($subjectAdmin);
        $template->method('getContentToAdmin')->willReturn($contentAdmin);
        $template->method('getSenderToUser')->willReturn($fromUser);
        $template->method('getRecipientToUser')->willReturn($toUser);
        $template->method('getReplyToUser')->willReturn(''); // todo
        $template->method('getCcToUser')->willReturn(''); // todo
        $template->method('getBccToUser')->willReturn(''); // todo
        $template->method('getSubjectToUser')->willReturn($subjectUser);
        $template->method('getContentToUser')->willReturn($contentUser);
        $template->method('hasTranslation')->willReturn(true); // todo: Test per quando la traduzione non esiste

        return $template;
    }

    /**
     * @return NotificationTemplateGeneratorInterface|MockObject
     */
    protected function createNotificationTemplateGeneratorMock()
    {
        return $this->createMock(NotificationTemplateGeneratorInterface::class);
    }

    /**
     * @return NotificationTemplateManagerInterface|MockObject
     */
    protected function createNotificationTemplateManagerMock(
        NotificationInterface $notification,
        NotificationTemplate $template
    ) {
        $notifications = [
            $notification->getName() => $template,
        ];

        $manager = $this->createMock(NotificationTemplateManagerInterface::class);

        $manager->method('findByNotification')->willReturnCallback(function ($name) use ($notifications) {
            return $notifications[$name];
        });

        return $manager;
    }

    /**
     * @return SiteInterface|MockObject
     */
    protected function createSiteMock()
    {
        $site = $this->createMock(SiteInterface::class);

        $site->method('getOption')->willReturnCallback(function (string $option) {
            $map = [
                'company_name'     => 'ACME',
                'company_type'     => 'Srl',
                'company_email'    => 'admin@website.com',
                'company_address'  => 'via Rossi, 123',
                'company_postcode' => '10100',
                'company_city'     => 'Rome',
                'company_province' => 'RM',
                'company_tel'      => '0123456789',
            ];

            if (!isset($map[$option])) {
                return $option;
            }

            return $map[$option];
        });

        $site->method('getName')->willReturn('ACME');

        $site->method('getThemesPath')->willReturnCallback(function (string $file) {
            return __DIR__ . '/Fixture/theme/' . $file;
        });

        $site->method('getThemesUrl')->willReturnCallback(function (string $file) {
            return 'https://example.com/theme/' . $file;
        });

        return $site;
    }

    /**
     * @return Translator|MockObject
     */
    protected function createTranslatorMock()
    {
        $translator = $this->createMock(Translator::class);

        $translator->method('trans')->willReturnCallback(function ($string) {
            return $string;
        });

        $translator->method('get')->willReturnCallback(function ($string) {
            return $string;
        });

        return $translator;
    }
}
