<?php

/*
 * This file is part of the pongho2 project.
 *
 * (c) Meta Line Srl
 */

namespace Application\Core\Tests\Mailer;

use Application\Core\Entity\SiteInterface;
use Application\Core\Mailer\Helper;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Pongho\Core\LocalizationInterface;
use Symfony\Component\Mime\Address;

class HelperTest extends TestCase
{
    private $helper;

    protected function setUp(): void
    {
        $this->helper = new Helper($this->getSiteMock(), $this->getLocalizationMock());
    }

    /**
     * @dataProvider addressesStringProvider
     */
    public function testExtractAddressesFromString(string $string, array $addresses): void
    {
        $this->assertEquals($addresses, $this->helper->extractAddressesFromString($string));
    }

    public function addressesStringProvider(): iterable
    {
        return [
            'empty string'          => [
                '',
                [],
            ],
            'email'                 => [
                'email@example.com',
                [
                    new Address('email@example.com'),
                ],
            ],
            'email and comma'       => [
                'email@example.com,',
                [
                    new Address('email@example.com'),
                ],
            ],
            'email and name'        => [
                'Name <email@example.com>',
                [
                    new Address('email@example.com', 'Name'),
                ],
            ],
            'multiple emails'       => [
                'email@example.com, email2@example.com, email3@example.com',
                [
                    new Address('email@example.com'),
                    new Address('email2@example.com'),
                    new Address('email3@example.com'),
                ],
            ],
            'multiple emails+names' => [
                'email@example.com, Name<email2@example.com>, Name3 <email3@example.com>',
                [
                    new Address('email@example.com'),
                    new Address('email2@example.com', 'Name'),
                    new Address('email3@example.com', 'Name3'),
                ],
            ],
            'with special chars'    => [
                'email@e_xample.com, My nàme<_email{2}@example.com>, Name_Surname <strange.email+address@example.com>',
                [
                    new Address('email@e_xample.com'),
                    new Address('_email{2}@example.com', 'My nàme'),
                    new Address('strange.email+address@example.com', 'Name_Surname'),
                ],
            ],
        ];
    }

    /**
     * @dataProvider replaceVarsProvider
     */
    public function testReplaceVars(string $text, array $vars, string $expected): void
    {
        $this->assertEquals($expected, $this->helper->replaceVars($text, $vars));
    }

    public function replaceVarsProvider(): iterable
    {
        return [
            'simple email'                      => [
                '%NAME% <%EMAIL%>',
                [
                    'NAME'  => 'Foo',
                    'EMAIL' => 'foo@example.com',
                ],
                'Foo <foo@example.com>',
            ],
            'uppercase and lowercase keys vars' => [
                '%name% <%EMAIL%>',
                [
                    'NAME'  => 'Foo',
                    'email' => 'foo@example.com',
                ],
                'Foo <foo@example.com>',
            ],
            'empty vars'                        => [
                'lorem ipsum dolor sit amet',
                [],
                'lorem ipsum dolor sit amet',
            ],
            'global variables'                  => [
                '{C_company_name} %C_company_name% {C_site_name} %C_site_name% {L_foo} %L_foo% {var} %var% {undefined_var} %undefined_var%',
                [
                    'var' => 'baz',
                ],
                'ACME ACME ACME Website ACME Website foo foo baz baz {undefined_var} %undefined_var%',
            ],
        ];
    }

    private function getSiteMock(): SiteInterface|MockObject
    {
        $mock = $this->createMock(SiteInterface::class);

        $mock->expects($this->any())->method('getName')->willReturn('ACME Website');
        $mock->expects($this->any())->method('getOption')->will(
            $this->returnCallback(
                function ($string) {
                    $options = [
                        'company_email' => 'company@example.com',
                        'company_name'  => 'ACME',
                    ];

                    return $options[$string] ?? null;
                }
            )
        );

        return $mock;
    }

    private function getLocalizationMock(): LocalizationInterface|MockObject
    {
        $mock = $this->createMock(LocalizationInterface::class);

        $mock->expects($this->any())->method('get')->will(
            $this->returnCallback(
                function ($string) {
                    return $string;
                }
            )
        );

        return $mock;
    }
}
