<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Tests\I18n;

use Application\Core\I18n\LocaleInterface;
use Application\Core\I18n\LocaleRegistry;

class LocaleRegistryTest extends \PHPUnit\Framework\TestCase
{
    public function testLocale(): void
    {
        $registry = new LocaleRegistry();

        $locale = $this->createMock(LocaleInterface::class);
        $registry->register('it_IT', function () use ($locale) {
            return $locale;
        });

        $otherLocale = $this->createMock(LocaleInterface::class);
        $registry->register('other', function () use ($otherLocale) {
            return $otherLocale;
        });

        $this->assertSame($locale, $registry->getLocale('it_IT'));
        $this->assertSame($otherLocale, $registry->getLocale('other'));
    }

    public function testShareLocale(): void
    {
        $registry = new LocaleRegistry();

        $registry->register('it_IT', function () {
            return $this->createMock(LocaleInterface::class);
        });

        $this->assertSame($registry->getLocale('it_IT'), $registry->getLocale('it_IT'));
    }

    public function testInternalLocale(): void
    {
        $registry = new LocaleRegistry();
        $this->assertInstanceOf(LocaleInterface::class, $registry->getLocale('it_IT'));
        $this->assertInstanceOf(LocaleInterface::class, $registry->getLocale('en_US'));
    }

    public function testDefaultLocale(): void
    {
        $locale = $this->createMock(LocaleInterface::class);

        $registry = new LocaleRegistry();

        $registry->registerDefault(function () use ($locale) {
            return $locale;
        });

        $this->assertSame($locale, $registry->getLocale('foo'));
    }

    public function testNoLocaleException(): void
    {
        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage('No Locale registered for "undefined".');
        (new LocaleRegistry())->getLocale('undefined');
    }

    public function testFactoryMustReturnValidLocale(): void
    {
        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage('The factory for the locale "it_IT" must be return an implementation of Application\Core\I18n\LocaleInterface');
        $registry = new LocaleRegistry();
        $registry->register('it_IT', function () {
            return 'invalid';
        });

        $registry->getLocale('it_IT');
    }

    public function testDefaultFactoryMustReturnValidLocale(): void
    {
        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage('The factory for the locale "xx_YY" must be return an implementation of Application\Core\I18n\LocaleInterface');
        $registry = new LocaleRegistry();
        $registry->registerDefault(function () {
            return 'invalid';
        });

        $registry->getLocale('xx_YY');
    }
}
