<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Tests\I18n\Country\Italy;

use Application\Core\I18n\Address\Address;
use Application\Core\I18n\Address\AddressInterface;
use Application\Core\I18n\Country\IT\AddressValidator;
use Application\Core\Model\Country;
use Application\Core\Model\Province;
use PHPUnit\Framework\MockObject\MockObject;

class AddressValidatorTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var AddressValidator
     */
    protected $validator;

    protected function setUp(): void
    {
        $this->validator = new AddressValidator();
    }

    /**
     *
     * @dataProvider provider
     */
    public function testValidate(AddressInterface $address, array $expectedErrors): void
    {
        $errors = [];
        $this->validator->validate($address, $errors);

        $this->assertEquals($expectedErrors, $errors);
    }

    public function provider()
    {
        return [
            [
                new Address(),
                [
                    'name'      => 'required_name',
                    'surname'   => 'required_surname',
                    'address'   => 'required_address',
                    'city'      => 'required_city',
                    'postcode'  => 'required_postcode',
                    'province'  => 'required_province',
                    'country'   => 'required_country',
                    'telephone' => 'required_telephone',
                ],
            ],
            [
                new Address('Paolino'),
                [
                    'surname'   => 'required_surname',
                    'address'   => 'required_address',
                    'city'      => 'required_city',
                    'postcode'  => 'required_postcode',
                    'province'  => 'required_province',
                    'country'   => 'required_country',
                    'telephone' => 'required_telephone',
                ],
            ],
            [
                new Address('Paolino', 'Paperino'),
                [
                    'address'   => 'required_address',
                    'city'      => 'required_city',
                    'postcode'  => 'required_postcode',
                    'province'  => 'required_province',
                    'country'   => 'required_country',
                    'telephone' => 'required_telephone',
                ],
            ],
            [
                new Address(
                    'Paolino',
                    'Paperino',
                    '',
                    'via Marconi, 56',
                    '',
                    '00123',
                    'Roma',
                    $this->getProvinceMock(),
                    $this->getCountryMock(),
                    '123456'
                ),
                [],
            ],
            [
                new Address(
                    'Paolino',
                    'Paperino',
                    '',
                    'via Marconi, 56',
                    '',
                    123,
                    'Roma',
                    $this->getProvinceMock(),
                    $this->getCountryMock(),
                    '123456'
                ),
                [
                    'postcode' => 'invalid_postcode_format',
                ],
            ],
        ];
    }

    /**
     * @return Province
     */
    protected function getProvinceMock(): Province|MockObject
    {
        $province = $this->getMockBuilder(Province::class)
            ->disableOriginalConstructor()
            ->setMethods(['__wakeup'])
            ->getMock();

        /**
         * @var Province $province
         */

        return $province;
    }

    /**
     * @return Country
     */
    protected function getCountryMock(): Country|MockObject
    {
        $country = $this->getMockBuilder(Country::class)
            ->disableOriginalConstructor()
            ->setMethods(['__wakeup'])
            ->getMock();

        /**
         * @var Country $country
         */

        return $country;
    }
}
