<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Model\NotificationTemplate;
use InvalidArgumentException;

class NotificationTemplateGenerator implements NotificationTemplateGeneratorInterface
{
    /**
     * @var NotificationConfig
     */
    protected $config;

    /**
     * @var LanguageInterface[]
     */
    protected $languages;

    /**
     * @param LanguageInterface[] $languages
     */
    public function __construct(array $languages, ?string $config_class = null)
    {
        if ($config_class !== null) {
            if (!class_exists($config_class)) {
                throw new InvalidArgumentException(
                    sprintf('The class "%s" for the notification config generator does not exists', $config_class)
                );
            }

            $config = new $config_class();
        } else {
            $config = new NotificationConfig();
        }

        $this->config = $config;
        $this->languages = $languages;
    }

    public function generate(string $name, string $path): NotificationTemplate
    {
        $this->config->prepare($name, $path);

        $template = new NotificationTemplate([
            'notification'       => $name,

            'sender_to_admin'    => $this->config->getSenderToAdmin(),
            'recipient_to_admin' => $this->config->getRecipientToAdmin(),
            'reply_to_admin'     => $this->config->getReplyToAdmin(),
            'cc_to_admin'        => $this->config->getCcToAdmin(),
            'bcc_to_admin'       => $this->config->getBccToAdmin(),
            'subject_to_admin'   => $this->config->getSubjectToAdmin(),
            'content_to_admin'   => $this->config->getContentToAdmin(),

            'sender_to_user'     => $this->config->getSenderToUser(),
            'recipient_to_user'  => $this->config->getRecipientToUser(),
            'reply_to_user'      => $this->config->getReplyToUser(),
            'cc_to_user'         => $this->config->getCcToUser(),
            'bcc_to_user'        => $this->config->getBccToUser(),
        ]);

        foreach ($this->languages as $language) {
            $template->prepareTranslation(
                $language->getId(),
                [
                    'subject_to_user' => $this->config->getSubjectToUser($language->getIso()),
                    'content_to_user' => $this->config->getContentToUser($language->getIso()),
                ]
            );
        }

        return $template;
    }
}
