<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

use Application\Core\Entity\SiteInterface;
use Application\Core\Mailer\Helper;
use Application\Core\Model\Manager\NotificationTemplateManagerInterface;
use Application\Core\Model\NotificationTemplate;
use Application\Core\Notification\Exception\SendException;
use Psr\Log\LoggerInterface;
use RuntimeException;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;
use Symfony\Component\Mime\Email;

class NotificationSender implements NotificationSenderInterface
{
    /**
     * @var Helper
     */
    protected $helper;

    /**
     * @var MailerInterface
     */
    protected $mailer;

    /**
     * @var NotificationTemplateManagerInterface
     */
    protected $template_manager;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var SiteInterface
     */
    protected $site;

    public function __construct(
        SiteInterface $site,
        MailerInterface $mailer,
        Helper $helper,
        NotificationTemplateManagerInterface $templateManager,
        LoggerInterface $logger
    ) {
        $this->site = $site;
        $this->helper = $helper;
        $this->mailer = $mailer;
        $this->template_manager = $templateManager;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function send(NotificationInterface $notification): void
    {
        $vars = $notification->getEmailVars();

        /** @var NotificationTemplate $notification_template */
        $notification_template = $this->template_manager->findByNotification(
            $notification->getName(),
            $notification->getLanguage()
        );

        if ($notification->getOptions()->canSendToUser()) {
            $from = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getSenderToUser(), $vars)
            );

            $to = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getRecipientToUser(), $vars)
            );

            $replyTo = [];
            if ($notification->getOptions()->canReplyToUser()) {
                $replyTo = $this->helper->extractAddressesFromString(
                    $this->helper->replaceVars($notification_template->getReplyToUser(), $vars)
                );
            }

            $cc = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getCcToUser(), $vars)
            );

            $bcc = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getBccToUser(), $vars)
            );

            try {
                $languageId = $notification->getLanguage()->getId();

                if (!$notification_template->hasTranslation($languageId)) {
                    $languageId = $this->site->getDefaultLanguageId();

                    $this->logger->debug(
                        '[Notification] Notification language not found, falling back to site default language',
                        [
                            'name'      => $notification->getName(),
                            'prev lang' => $notification->getLanguage()->getId(),
                            'new lang'  => $languageId,
                        ]
                    );
                }

                if (!$notification_template->hasTranslation($languageId)) {
                    $this->logger->error(
                        '[Notification] Unable to find a suitable fall back language for the notification',
                        [
                            'name'     => $notification->getName(),
                            'language' => $notification->getLanguage()->getId(),
                        ]
                    );

                    throw new RuntimeException('Unable to find a suitable fall back language for the notification, the web admin has been already notified');
                }

                $email = (new Email())
                    ->from(...$from)
                    ->to(...$to)
                    ->replyTo(...$replyTo)
                    ->cc(...$cc)
                    ->bcc(...$bcc)
                    ->subject($this->helper->replaceVars($notification_template->getSubjectToUser($languageId), $vars))
                    ->html($this->helper->replaceVars($notification_template->getContentToUser($languageId), $vars));

                $this->mailer->send($this->helper->prepareEmail($email));

                $this->logger->info(
                    '[Notification] Invio della notifica "' . $notification->getName() . '" all’utente',
                    [
                        'from'    => $this->prepareAddressesForLog($from),
                        'to'      => $this->prepareAddressesForLog($to),
                        'replyTo' => $this->prepareAddressesForLog($replyTo),
                        'cc'      => $this->prepareAddressesForLog($cc),
                        'bcc'     => $this->prepareAddressesForLog($bcc),
                    ]
                );
            } catch (TransportExceptionInterface $e) {
                $this->logger->error(
                    '[Notification] Errore invio della notifica "' . $notification->getName() . '" all’utente',
                    [
                        'from'      => $this->prepareAddressesForLog($from),
                        'to'        => $this->prepareAddressesForLog($to),
                        'replyTo'   => $this->prepareAddressesForLog($replyTo),
                        'cc'        => $this->prepareAddressesForLog($cc),
                        'bcc'       => $this->prepareAddressesForLog($bcc),
                        'exception' => $e,
                    ]
                );

                throw new SendException($e->getMessage(), 'user', $e);
            }
        }

        if ($notification->getOptions()->canSendToAdmin()) {
            $from = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getSenderToAdmin(), $vars)
            );

            $to = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getRecipientToAdmin(), $vars)
            );

            $replyTo = [];
            if ($notification->getOptions()->canReplyToAdmin()) {
                $replyTo = $this->helper->extractAddressesFromString(
                    $this->helper->replaceVars($notification_template->getReplyToAdmin(), $vars)
                );
            }

            $cc = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getCcToAdmin(), $vars)
            );

            $bcc = $this->helper->extractAddressesFromString(
                $this->helper->replaceVars($notification_template->getBccToAdmin(), $vars)
            );

            try {
                $email = (new Email())
                    ->from(...$from)
                    ->to(...$to)
                    ->replyTo(...$replyTo)
                    ->cc(...$cc)
                    ->bcc(...$bcc)
                    ->subject($this->helper->replaceVars($notification_template->getSubjectToAdmin(), $vars))
                    ->html($this->helper->replaceVars($notification_template->getContentToAdmin(), $vars));

                $this->mailer->send($this->helper->prepareEmail($email));

                $this->logger->info(
                    '[Notification] Invio della notifica "' . $notification->getName() . '" all’amministratore',
                    [
                        'from'    => $this->prepareAddressesForLog($from),
                        'to'      => $this->prepareAddressesForLog($to),
                        'replyTo' => $this->prepareAddressesForLog($replyTo),
                        'cc'      => $this->prepareAddressesForLog($cc),
                        'bcc'     => $this->prepareAddressesForLog($bcc),
                    ]
                );
            } catch (TransportExceptionInterface $e) {
                $this->logger->error(
                    '[Notification] Errore invio della notifica "' . $notification->getName() . '" all’amministratore',
                    [
                        'from'      => $this->prepareAddressesForLog($from),
                        'to'        => $this->prepareAddressesForLog($to),
                        'replyTo'   => $this->prepareAddressesForLog($replyTo),
                        'cc'        => $this->prepareAddressesForLog($cc),
                        'bcc'       => $this->prepareAddressesForLog($bcc),
                        'exception' => $e,
                    ]
                );

                throw new SendException($e->getMessage(), 'admin', $e);
            }
        }
    }

    /**
     * @param Address[] $addresses
     */
    public function prepareAddressesForLog(array $addresses): array
    {
        $new = [];
        foreach ($addresses as $address) {
            $new[] = $address->toString();
        }

        return $new;
    }
}
