<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

class NotificationOptions
{
    /**
     * @var bool
     */
    protected $can_send_to_admin;

    /**
     * @var bool
     */
    protected $can_send_to_user;

    /**
     * @var bool
     */
    protected $can_change_sender_to_admin;

    /**
     * @var bool
     */
    protected $can_change_sender_to_user;

    /**
     * @var bool
     */
    protected $can_change_recipient_to_admin;

    /**
     * @var bool
     */
    protected $can_change_recipient_to_user;

    /**
     * @var bool
     */
    protected $can_reply_to_admin;

    /**
     * @var bool
     */
    protected $can_reply_to_user;

    /**
     * @var bool
     */
    protected $can_change_reply_to_admin;

    /**
     * @var bool
     */
    protected $can_change_reply_to_user;

    /**
     * @param bool|array $send_to_admin             Se la notifica va inviata all’amministratore.
     * @param bool       $send_to_user              Se la notifica va inviata all’utente.
     * @param bool       $change_sender_to_admin    Se è possibile modificare da interfaccia il mittente per l’admin.
     * @param bool       $change_sender_to_user     Se è possibile modificare da interfaccia il mittente per l’utente.
     * @param bool       $change_recipient_to_admin Se è possibile modificare da interfaccia il destinatario per l’admin.
     * @param bool       $change_recipient_to_user  Se è possibile modificare da interfaccia il destinatario per l’utente.
     * @param bool       $reply_to_admin            Se deve essere inserita l’intestazione "Reply-To" alla notifica per l’admin.
     * @param bool       $reply_to_user             Se deve essere inserita l’intestazione "Reply-To" alla notifica per l’utente.
     * @param bool       $change_reply_to_admin     Se è possibile modificare da interfaccia il "Reply-To" per l’admin.
     * @param bool       $change_reply_to_user      Se è possibile modificare da interfaccia il "Reply-To" per l’utente.
     */
    public function __construct(
        $send_to_admin = [],
        ?bool $send_to_user = null,
        ?bool $change_sender_to_admin = null,
        ?bool $change_sender_to_user = null,
        ?bool $change_recipient_to_admin = null,
        ?bool $change_recipient_to_user = null,
        bool $reply_to_admin = false,
        bool $reply_to_user = false,
        bool $change_reply_to_admin = false,
        bool $change_reply_to_user = false
    ) {
        if (is_array($send_to_admin)) {
            $this->parseOptions($send_to_admin);
        } else {
            $this->can_send_to_admin = $send_to_admin;
            $this->can_send_to_user = $send_to_user;
            $this->can_change_sender_to_admin = $change_sender_to_admin;
            $this->can_change_sender_to_user = $change_sender_to_user;
            $this->can_change_recipient_to_admin = $change_recipient_to_admin;
            $this->can_change_recipient_to_user = $change_recipient_to_user;
            $this->can_reply_to_admin = $reply_to_admin;
            $this->can_reply_to_user = $reply_to_user;
            $this->can_change_reply_to_admin = $change_reply_to_admin;
            $this->can_change_reply_to_user = $change_reply_to_user;

            trigger_error('NotificationOptions::__construct() deprecation.', E_USER_DEPRECATED);
        }
    }

    private function parseOptions(array $options = []): void
    {
        // admin
        $this->can_send_to_admin = isset($options['send_to_admin']) && (bool) $options['send_to_admin'];

        if (isset($options['can_change_sender_to_admin'])) {
            $this->can_change_sender_to_admin = (bool) $options['can_change_sender_to_admin'];
        } else {
            $this->can_change_sender_to_admin = $this->can_send_to_admin;
        }

        if (isset($options['can_change_recipient_to_admin'])) {
            $this->can_change_recipient_to_admin = (bool) $options['can_change_recipient_to_admin'];
        } else {
            $this->can_change_recipient_to_admin = $this->can_send_to_admin;
        }

        if (isset($options['can_reply_to_admin'])) {
            $this->can_reply_to_admin = (bool) $options['can_reply_to_admin'];
        } else {
            $this->can_reply_to_admin = $this->can_send_to_admin;
        }

        if (isset($options['can_change_reply_to_admin'])) {
            $this->can_change_reply_to_admin = (bool) $options['can_change_reply_to_admin'];
        } else {
            $this->can_change_reply_to_admin = false;
        }

        // user
        $this->can_send_to_user = isset($options['send_to_user']) && (bool) $options['send_to_user'];

        if (isset($options['can_change_sender_to_user'])) {
            $this->can_change_sender_to_user = (bool) $options['can_change_sender_to_user'];
        } else {
            $this->can_change_sender_to_user = $this->can_send_to_user;
        }

        if (isset($options['can_change_recipient_to_user'])) {
            $this->can_change_recipient_to_user = (bool) $options['can_change_recipient_to_user'];
        } else {
            $this->can_change_recipient_to_user = false;
        }

        $this->can_reply_to_user = isset($options['can_reply_to_user']) && (bool) $options['can_reply_to_user'];

        if (isset($options['can_change_reply_to_user'])) {
            $this->can_change_reply_to_user = (bool) $options['can_change_reply_to_user'];
        } else {
            $this->can_change_reply_to_user = $this->can_reply_to_user;
        }
    }

    public function canSendToAdmin(): bool
    {
        return $this->can_send_to_admin;
    }

    public function canSendToUser(): bool
    {
        return $this->can_send_to_user;
    }

    public function canChangeRecipientToAdmin(): bool
    {
        return $this->can_change_recipient_to_admin;
    }

    public function canChangeRecipientToUser(): bool
    {
        return $this->can_change_recipient_to_user;
    }

    public function canChangeSenderToAdmin(): bool
    {
        return $this->can_change_sender_to_admin;
    }

    public function canChangeSenderToUser(): bool
    {
        return $this->can_change_sender_to_user;
    }

    public function canReplyToAdmin(): bool
    {
        return $this->can_reply_to_admin;
    }

    public function canReplyToUser(): bool
    {
        return $this->can_reply_to_user;
    }

    public function canChangeReplyToAdmin(): bool
    {
        return $this->can_change_reply_to_admin;
    }

    public function canChangeReplyToUser(): bool
    {
        return $this->can_change_reply_to_user;
    }
}
