<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

use BadMethodCallException;
use LogicException;

class NotificationConfig implements NotificationConfigInterface
{
    /**
     * @var string
     */
    protected $path;

    /**
     * @var object
     */
    protected $config;

    /**
     * @var array
     */
    protected $translations;

    /**
     * @var bool
     */
    protected $prepared = false;

    public function prepare(string $notification_name, string $path): void
    {
        $this->path = rtrim($path, '/') . '/';
        $filename = $this->path . $notification_name . '.json';

        if (!file_exists($filename)) {
            throw new \InvalidArgumentException('The file "' . $filename . '" does not exists');
        }

        $this->config = json_decode(file_get_contents($filename));

        if (json_last_error()) {
            throw new LogicException(
                sprintf(
                    'JSON decode of the notification "%s" returns an error: %s',
                    $notification_name,
                    json_last_error_msg()
                )
            );
        }

        $this->prepareTranslations();

        $this->prepared = true;
    }

    /**
     * Imposta le traduzioni in un modo più semplice da gestire.
     */
    protected function prepareTranslations()
    {
        $this->translations = [];

        if (isset($this->config->translations)) {
            foreach ($this->config->translations as $translation) {
                $this->translations[$translation->language] = $translation->fields;
            }
        }
    }

    public function getSenderToAdmin(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->sender_to_admin) && !empty($this->config->sender_to_admin)) {
            return $this->config->sender_to_admin;
        }

        return '';
    }

    public function getRecipientToAdmin(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->recipient_to_admin) && !empty($this->config->recipient_to_admin)) {
            return $this->config->recipient_to_admin;
        }

        return '';
    }

    public function getReplyToAdmin(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->reply_to_admin) && !empty($this->config->reply_to_admin)) {
            return $this->config->reply_to_admin;
        }

        return '';
    }

    public function getCcToAdmin(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->cc_to_admin) && !empty($this->config->cc_to_admin)) {
            return $this->config->cc_to_admin;
        }

        return '';
    }

    public function getBccToAdmin(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->bcc_to_admin) && !empty($this->config->bcc_to_admin)) {
            return $this->config->bcc_to_admin;
        }

        return '';
    }

    public function getSenderToUser(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->sender_to_user) && !empty($this->config->sender_to_user)) {
            return $this->config->sender_to_user;
        }

        return '';
    }

    public function getRecipientToUser(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->recipient_to_user) && !empty($this->config->recipient_to_user)) {
            return $this->config->recipient_to_user;
        }

        return '';
    }

    public function getReplyToUser(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->reply_to_user) && !empty($this->config->reply_to_user)) {
            return $this->config->reply_to_user;
        }

        return '';
    }

    public function getCcToUser(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->cc_to_user) && !empty($this->config->cc_to_user)) {
            return $this->config->cc_to_user;
        }

        return '';
    }

    public function getBccToUser(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->bcc_to_user) && !empty($this->config->bcc_to_user)) {
            return $this->config->bcc_to_user;
        }

        return '';
    }

    public function getSubjectToAdmin(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->subject_to_admin) && !empty($this->config->subject_to_admin)) {
            return $this->config->subject_to_admin;
        }

        return '';
    }

    public function getContentToAdmin(): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->content_to_admin) && !empty($this->config->content_to_admin)) {
            $filename = $this->path . $this->config->content_to_admin;

            return $this->loadFileContent($filename);
        }

        return '';
    }

    public function getSubjectToUser(string $language_iso): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (
            isset($this->translations[$language_iso]->subject_to_user)
            && !empty($this->translations[$language_iso]->subject_to_user)
        ) {
            return $this->translations[$language_iso]->subject_to_user;
        }

        if (
            isset($this->translations['it']->subject_to_user)
            && !empty($this->translations['it']->subject_to_user)
        ) {
            return $this->translations['it']->subject_to_user;
        }

        return '';
    }

    public function getContentToUser(string $language_iso): string
    {
        if (!$this->prepared) {
            throw new BadMethodCallException('The config must be prepared to get its properties');
        }

        if (
            isset($this->translations[$language_iso]->content_to_user)
            && !empty($this->translations[$language_iso]->content_to_user)
        ) {
            $filename = $this->path . $this->translations[$language_iso]->content_to_user;
            $content = $this->loadFileContent($filename);

            if ($content) {
                return $content;
            }
        }

        if (
            isset($this->translations['it']->content_to_user)
            && !empty($this->translations['it']->content_to_user)
        ) {
            return $this->loadFileContent($this->path . $this->translations['it']->content_to_user);
        }

        return '';
    }

    protected function loadFileContent(string $filename): string
    {
        if (file_exists($filename) && is_file($filename)) {
            return file_get_contents($filename);
        }

        return '';
    }
}
