<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

use Application\Core\Entity\NotificationTemplateInterface;
use Application\Core\Entity\SiteInterface;
use Application\Core\I18n\Translation\Translator;

abstract class AbstractNotification implements NotificationInterface
{
    /**
     * @var NotificationOptions
     */
    protected $options;

    /**
     * @var SiteInterface
     */
    protected $site;

    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var NotificationTemplateGeneratorInterface
     */
    protected $template_generator;

    public function __construct(SiteInterface $site, Translator $translator, NotificationTemplateGeneratorInterface $template_generator)
    {
        $this->site = $site;
        $this->translator = $translator;
        $this->template_generator = $template_generator;
    }

    /**
     * {@inheritdoc}
     */
    public function getDefaultTemplate(): NotificationTemplateInterface
    {
        return $this->template_generator->generate($this->getName(), $this->getDefaultTemplatePath());
    }

    protected function getDefaultTemplatePath(): string
    {
        // Uso le Reflection per riuscire a ricavare il percorso delle classi fuori dall'applicazione Core
        // perché usando __DIR__ risulterebbe sempre il percorso di questa stessa classe e non di quella che la eredita
        $reflection_class = new \ReflectionClass($this);
        $path = dirname($reflection_class->getFileName());

        return $path . '/Resources';
    }

    /**
     * {@inheritdoc}
     */
    public function getVars(): array
    {
        return [
            'PONGHO_SITE_NAME',
            'PONGHO_SITE_URL',
            'PONGHO_COMPANY_SITE',
            'PONGHO_COMPANY_LOGO',
            'PONGHO_COMPANY_NAME',
            'PONGHO_COMPANY_TYPE',
            'PONGHO_COMPANY_EMAIL',
            'PONGHO_COMPANY_ADDRESS',
            'PONGHO_COMPANY_POSTCODE',
            'PONGHO_COMPANY_CITY',
            'PONGHO_COMPANY_PROVINCE',
            'PONGHO_COMPANY_TEL',
            'EMAIL_SEND_DATE',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function getEmailVars(): array
    {
        return [
            'PONGHO_SITE_NAME'        => $this->site->getName(),
            'PONGHO_SITE_URL'         => $this->site->getDomain(),
            'PONGHO_COMPANY_SITE'     => $this->site->getDomain(),
            'PONGHO_COMPANY_LOGO'     => $this->getLogoUrl(),
            'PONGHO_COMPANY_NAME'     => $this->site->getOption('company_name'),
            'PONGHO_COMPANY_TYPE'     => $this->site->getOption('company_type'),
            'PONGHO_COMPANY_EMAIL'    => $this->site->getOption('company_email'),
            'PONGHO_COMPANY_ADDRESS'  => $this->site->getOption('company_address'),
            'PONGHO_COMPANY_POSTCODE' => $this->site->getOption('company_postcode'),
            'PONGHO_COMPANY_CITY'     => $this->site->getOption('company_city'),
            'PONGHO_COMPANY_PROVINCE' => $this->site->getOption('company_province'),
            'PONGHO_COMPANY_TEL'      => $this->site->getOption('company_tel'),
            'EMAIL_SEND_DATE'         => date('d/m/Y'),
        ];
    }

    public function getLogoUrl(): string
    {
        $logo_path = $this->site->getThemesPath('images/logo-email.png');

        if (!file_exists($logo_path)) {
            throw new \RuntimeException(sprintf('Logo not found on path "%s"', $logo_path));
        }

        return $this->site->getThemesUrl('images/logo-email.png');
    }
}
