<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\SettingsInterface;

/**
 * Modello base per moduli e siti che necessitano di impostazioni.
 *
 * @property string $settings
 */
abstract class Settings extends Base implements SettingsInterface
{
    /**
     * @var array
     */
    protected $_settings;

    /**
     * Callback 'after_construct'.
     *
     * @var array
     */
    public static $after_construct = ['prepareSettingsOnCreate'];

    /**
     * Callback 'before_save'.
     *
     * @var array
     */
    public static $before_save = ['prepareSettingsOnSave'];

    /**
     * Prepara le opzioni per la lettura.
     */
    public function prepareSettingsOnCreate()
    {
        $settings = is_string($this->settings) ? unserialize($this->settings) : [];
        $settings = is_array($settings) ? $settings : [];

        $this->_settings = array_merge($this->getDefaultOptions(), $settings);
    }

    /**
     * Prepara le opzioni per il salvataggio in database.
     */
    public function prepareSettingsOnSave()
    {
        $this->settings = serialize($this->_settings);
    }

    /**
     * Imposta le opzioni di default.
     *
     * @return $this
     */
    public function setDefaultOptions(array $options)
    {
        foreach ($options as $name => $value) {
            if (!array_key_exists($name, $this->_settings)) {
                $this->_settings[$name] = $value;
            }
        }

        return $this;
    }

    /**
     * Indica se l’opzione esiste.
     *
     * @param string $name
     * @return bool
     */
    public function hasOption($name)
    {
        return array_key_exists($name, $this->_settings);
    }

    /**
     * Restituisce una opzione.
     *
     * @param string $name
     * @param mixed  $default
     * @return mixed
     */
    public function getOption($name, $default = null)
    {
        return array_key_exists($name, $this->_settings) ? $this->_settings[$name] : $default;
    }

    /**
     * Imposta una opzione.
     *
     * @param string $name
     * @param mixed  $value
     */
    public function setOption($name, $value)
    {
        $this->_settings[$name] = $value;
    }

    /**
     * Elimina una opzione.
     *
     * @param string $name
     */
    public function delOption($name)
    {
        unset($this->_settings[$name]);
    }

    /**
     * Restituisce tutte le opzioni.
     *
     * @return array
     */
    public function allOptions()
    {
        return $this->_settings;
    }

    /**
     * Restituisce le opzioni di default.
     *
     * @return array
     */
    protected function getDefaultOptions()
    {
        return [];
    }
}
