<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\RoleInterface;

/**
 * Modello per i ruoli.
 *
 * @property int    $id
 * @property string $name
 * @property string $description
 * @property int    $position
 */
class Role extends Base implements RoleInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'roles';

    /**
     * Indica l’ID del ruolo speciale “Utente non registrato”.
     */
    public const USER_NOT_LOGGED = 1;

    /**
     * Indica l’ID del ruolo speciale “Utente registrato”.
     */
    public const USER_LOGGED = 2;

    /**
     * Indica l’ID del ruolo speciale “Amministratore”.
     */
    public const ADMIN = 3;

    /**
     * Lista dei ruoli speciali.
     *
     * @var array
     */
    public static $SPECIAL_ROLES = [self::USER_NOT_LOGGED, self::USER_LOGGED, self::ADMIN];

    /**
     * @var Permit[]
     */
    protected $permits;

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * {@inheritdoc}
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Indica se il ruolo è eliminabile.
     *
     * @return bool
     */
    public function isDeletable()
    {
        return !in_array($this->id, static::$SPECIAL_ROLES);
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        if (!$this->isDeletable()) {
            return true;
        }

        return $this->delPermits() && parent::delete();
    }

    /**
     * {@inheritdoc}
     */
    public function getPermits()
    {
        if ($this->permits === null) {
            $options = [
                'joins'      => 'LEFT JOIN ' . PermitRole::tableName() . ' AS pr ON pr.permit_id = `from`.id',
                'conditions' => [
                    'pr.role_id = ? AND `from`.is_enabled = ?',
                    $this->id,
                    true,
                ],
            ];

            $this->permits = [];
            foreach (Permit::all($options) as $permit) {
                $this->permits[] = $permit;
            }
        }

        return $this->permits;
    }

    /**
     * {@inheritdoc}
     */
    public function getPermitsKeys()
    {
        return array_map(
            function ($permit) {
                return $permit->key;
            },
            $this->getPermits()
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getPermitsIds()
    {
        return array_map(
            function ($permit) {
                return $permit->id;
            },
            $this->getPermits()
        );
    }

    /**
     * {@inheritdoc}
     */
    public function hasPermit($key)
    {
        if (str_ends_with($key, '.')) {
            $length = strlen($key);

            foreach ($this->getPermits() as $permit) {
                if (substr((string) $permit->key, 0, $length) === $key) {
                    return true;
                }
            }
        }

        foreach ($this->getPermits() as $permit) {
            if ($permit->key === $key) {
                return true;
            }
        }

        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function setPermits(array $permits)
    {
        // preferisco lavorare con gli ID dei permessi
        if (reset($permits) instanceof Permit) {
            foreach ($permits as &$permit) {
                $permit = $permit->id;
            }
        }

        $permits = array_unique($permits);

        // preferisco operare eseguendo una transazione
        $role = $this;

        self::transaction(
            function () use ($role, $permits): void {
                // rimuovo tutti i permessi
                $role->delPermits();

                foreach ($permits as $permitId) {
                    PermitRole::create([
                        'permit_id' => $permitId,
                        'role_id'   => $role->id,
                    ]);
                }
            }
        );

        $this->permits = null;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function addPermit($permitOrPermitKey)
    {
        if ($permitOrPermitKey instanceof Permit) {
            $permit = $permitOrPermitKey;
        } else {
            $permit = Permit::findOrCreate($permitOrPermitKey);
        }

        $permitRole = PermitRole::find([$permit->id, $this->getId()]);

        if (!$permitRole) {
            PermitRole::create([
                'permit_id' => $permit->id,
                'role_id'   => $this->id,
            ]);
        }

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function removePermit($permitOrPermitKey)
    {
        if ($permitOrPermitKey instanceof Permit) {
            $permit = $permitOrPermitKey;
        } else {
            $permit = Permit::findOrCreate($permitOrPermitKey);
        }

        $permitRole = PermitRole::find([$permit->id, $this->getId()]);

        if ($permitRole) {
            $permitRole->delete();
        }

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function delPermits()
    {
        $permits = PermitRole::all([
            'conditions' => ['role_id = ?', $this->id],
        ]);

        foreach ($permits as $permit) {
            $permit->delete();
        }

        $this->permits = null;

        return $this;
    }
}
