<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\ModuleSiteInterface;
use Pongho\Core\Kernel;

/**
 * Modello per le relazioni tra moduli e siti.
 *
 * @property int  $id
 * @property int  $module_id
 * @property int  $site_id
 * @property bool $is_enabled
 */
class ModuleSite extends Settings implements ModuleSiteInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'modules_sites';

    /**
     * {@inheritdoc}
     */
    public static function getDefaultModelOptions()
    {
        return [
            'select' => 'm.*, lm.*, `from`.*, lm.is_enabled AS language_enabled, `from`.is_enabled AS module_enabled, ls.language_id',
            'joins'  => 'INNER JOIN ' . Module::tableName() . ' AS m ON m.id = `from`.module_id'
                     . ' INNER JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.module_id = `from`.module_id'
                     . ' INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = lm.language_site_id AND ls.site_id = `from`.site_id',
        ];
    }

    /**
     * Restituisce il `ModuleSite` in base a `$module_id` e `$site_id` indicati.
     *
     * È necessario anche `$language_id` per conoscere il nome ed il percorso del modulo.
     *
     * @param string $module_id
     * @param int    $site_id
     * @param int    $language_id
     * @return ModuleSite
     */
    public static function findById($module_id, $site_id, $language_id)
    {
        $options = self::addCondition(self::getDefaultModelOptions(), [
            'm.id = :module AND ls.language_id = :language AND `from`.site_id = :site',
            'module'   => $module_id,
            'site'     => $site_id,
            'language' => $language_id,
        ]);

        return self::first($options);
    }

    /**
     * @param int  $site_id
     * @param int  $language_id
     * @param bool $only_enabled
     * @return ModuleSite[]
     */
    public static function findAll($site_id, $language_id, $only_enabled = true)
    {
        $options = self::addCondition(self::getDefaultModelOptions(), [
            'ls.language_id = :language AND `from`.site_id = :site AND `from`.is_enabled = :enabled',
            'site'     => $site_id,
            'language' => $language_id,
            'enabled'  => true,
        ]);

        if ($only_enabled) {
            $options = self::addCondition($options, [
                'lm.is_enabled = :language_enabled',
                'language_enabled' => true,
            ]);
        }

        return self::all($options);
    }

    /**
     * Restituisce le lingue associate al modulo.
     *
     * La lista restituita contiene modelli di tipo `Application\Core\Model\Language`.
     *
     * @return Language[]
     */
    public function getLanguages()
    {
        return Language::all([
            'select'     => 'ls.*, `from`.*',
            'joins'      => implode(' ', [
                'INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.language_id = `from`.id',
                'INNER JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.language_site_id = ls.id',
            ]),
            'conditions' => [
                'ls.site_id = :site AND lm.module_id = :module AND ls.is_enabled = :enabled AND lm.is_enabled = :enabled',
                'site'    => $this->site_id,
                'module'  => $this->module_id,
                'enabled' => true,
            ],
            'order'      => '`from`.id',
        ]);
    }

    /**
     * Restituisce il modulo associato.
     */
    public function getModule()
    {
        /** @var Manager\ModuleManager $manager */
        $manager = Kernel::instance()->getContainer()->get('module_manager');

        return $manager->findById($this->module_id);
    }

    /**
     * Restituisce il sito associato.
     */
    public function getSite()
    {
        /** @var Manager\SiteManager $manager */
        $manager = Kernel::instance()->getContainer()->get('site_manager');

        return $manager->findById($this->site_id);
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getModuleId()
    {
        return $this->module_id;
    }

    /**
     * @return int
     */
    public function getSiteId()
    {
        return $this->site_id;
    }

    /**
     * @return bool
     */
    public function isEnabled()
    {
        return $this->is_enabled;
    }

    /**
     * @return \Application\Core\Entity\LanguageModuleInterface[]
     */
    public function getLanguageModules()
    {
        /** @var Manager\LanguageModuleManagerInterface $language_module_manager */
        $language_module_manager = Kernel::instance()->getContainer()->get('language_module_manager');

        return $language_module_manager->findAllBySiteAndModule($this->site_id, $this->module_id);
    }

    /**
     * @param int|LanguageInterface $language
     * @return \Application\Core\Entity\LanguageModuleInterface
     */
    public function getLanguageModule($language = null)
    {
        if (is_numeric($language)) {
            $language_id = intval($language);
        } elseif ($language instanceof LanguageInterface) {
            $language_id = $language->getId();
        } elseif ($language === null) {
            $language_id = Kernel::instance()->getContainer()->get('language_site')->getLanguageId();
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'Language must be an integer or an instance of LanguageInterface. "%s" given instead.',
                    var_to_string($language)
                )
            );
        }

        /** @var Manager\LanguageModuleManagerInterface $language_module_manager */
        $language_module_manager = Kernel::instance()->getContainer()->get('language_module_manager');

        return $language_module_manager->findByLanguageAndModuleSite($language_id, $this->id);
    }

    /**
     * @param int|LanguageInterface $language
     * @return string
     */
    public function getName($language = null)
    {
        $language_module = $this->getLanguageModule($language);

        return $language_module ? $language_module->getName() : '';
    }

    /**
     * @param int|LanguageInterface $language
     * @return string
     */
    public function getPermalink($language = null)
    {
        $language_module = $this->getLanguageModule($language);

        return $language_module ? absolute_url($this->getLanguageModule($language)->getPath()) : '';
    }
}
