<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\SiteInterface;
use Application\Core\Model\NotificationTemplate;
use Application\Core\Model\NotificationTemplateTranslation;
use Application\Core\Notification\NotificationCollection;

class NotificationTemplateManager implements NotificationTemplateManagerInterface
{
    /**
     * @var \Application\Core\Entity\NotificationTemplateInterface[]
     */
    protected $models = [];

    /**
     * @var SiteInterface
     */
    protected $site;

    /**
     * @var NotificationCollection
     */
    protected $collection;

    public function __construct(SiteInterface $site, NotificationCollection $collection)
    {
        $this->site = $site;
        $this->collection = $collection;
    }

    /**
     * {@inheritdoc}
     */
    public function findById($id, ?LanguageInterface $language = null)
    {
        $conditions = [];

        if ($language instanceof LanguageInterface) {
            $conditions = [
                'ntt.language_id = :language',
                'language' => $language->getId(),
            ];
        }

        $options = $this->getDefaultQueryOptions($conditions);
        $options = NotificationTemplate::addCondition($options, ['id = :id', 'id' => $id]);

        // TODO: cache
        return NotificationTemplate::first($options);
    }

    /**
     * {@inheritdoc}
     */
    public function findByNotification($notificationName, ?LanguageInterface $language = null)
    {
        if (!array_key_exists($notificationName, $this->models)) {
            $notification = $this->collection->get($notificationName);

            $conditions = ['notification = :name', 'name' => $notificationName];

            if ($language && $notification->getOptions()->canSendToUser()) {
                $conditions[0] .= ' AND ntt.language_id = :language';
                $conditions['language'] = $language->getId();
            }

            $options = $this->getDefaultQueryOptions($conditions);

            $model = NotificationTemplate::first($options);

            if (!$model) {
                $model = $notification->getDefaultTemplate();
            }

            $this->models[$notificationName] = $model;
        }

        return $this->models[$notificationName];
    }

    /**
     * {@inheritdoc}
     */
    public function findAll(array $conditions = [], $orderBy = null, $limit = null, $offset = null)
    {
        $models = [];

        /** @var \Application\Core\Notification\NotificationInterface $notification */
        foreach ($this->collection as $notification) {
            $models[$notification->getName()] = new NotificationTemplate([
                'notification' => $notification->getName(),
            ]);
        }

        $options = $this->getDefaultQueryOptions($conditions, $orderBy, $limit, $offset);

        foreach ($this->query($options) as $model) {
            $models[$model->getNotificationName()] = $model;
        }

        return $this->getIdentityMap($models);
    }

    /**
     * {@inheritdoc}
     */
    public function countAll(array $conditions = [])
    {
        $options = $this->getDefaultQueryOptions($conditions);

        return NotificationTemplate::count($options);
    }

    /**
     * @param array $options
     * @return \Application\Core\Entity\NotificationTemplateInterface[]
     */
    protected function query($options)
    {
        return NotificationTemplate::all($options);
    }

    /**
     * @param \Application\Core\Entity\NotificationTemplateInterface[] $records
     * @return \Application\Core\Entity\NotificationTemplateInterface[]
     */
    protected function getIdentityMap(array $records)
    {
        $models = [];

        foreach ($records as $model) {
            if (!array_key_exists($model->getNotificationName(), $this->models)) {
                $this->models[$model->getNotificationName()] = $model;
                $models[] = $model;
            } else {
                $models[] = $this->models[$model->getNotificationName()];
            }
        }

        return $models;
    }

    /**
     * @param string $orderBy
     * @param int    $limit
     * @param int    $offset
     * @return array
     */
    protected function getDefaultQueryOptions(array $conditions = [], $orderBy = null, $limit = null, $offset = null)
    {
        $options = [
            'select' => 'ntt.*, `from`.*',
            'joins'  => 'LEFT JOIN ' . NotificationTemplateTranslation::tableName() . ' AS ntt ON ntt.template_id = `from`.id',
        ];

        $options = NotificationTemplate::addCondition($options, $conditions);
        $options = NotificationTemplate::addCondition(
            $options,
            ['`from`.site_id = :site', 'site' => $this->site->getId()]
        );

        if ($orderBy) {
            $options['order'] = $orderBy;
        }

        if ($limit) {
            $options['limit'] = $limit;
        }

        if ($offset) {
            $options['offset'] = $offset;
        }

        return $options;
    }
}
