<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\ModuleInterface;
use Application\Core\Model\Module;

class ModuleManager extends AbstractCoreManager implements ModuleManagerInterface
{
    /**
     * @var string
     */
    public static $model = 'Module';

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function (ModuleInterface $model) use ($site_id) {
                $module_site = $this->registry->getModuleSiteManager()->findBySiteAndModule($site_id, $model->getId());

                return $module_site !== null;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByController($controller)
    {
        return array_filter(
            $this->findAll(),
            function (ModuleInterface $model) use ($controller) {
                return $model->getController() === $controller;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findById($module_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $module_id);

        if (!$model) {
            throw new \InvalidArgumentException(sprintf('The module "%s" is undefined', $module_id));
        }

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function create($controller)
    {
        $model = new Module();

        $model->controller = $controller;

        return $model;
    }

    /**
     * @param ModuleInterface|Module $module
     * {@inheritdoc}
     */
    public function save(ModuleInterface $module)
    {
        if (!$module->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if (!$this->getCache()->isEmpty(static::$model)) {
            return;
        }

        /** @var Module $model */
        foreach (Module::all() as $model) {
            $this->getCache()->set(static::$model, $model->getId(), $model);
        }
    }
}
