<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\SiteInterface;
use Application\Core\Model\Application;
use Application\Core\Model\ApplicationSite;
use Application\Core\Model\Manager\Exception\ApplicationManagerNotFoundException;

class ApplicationManager implements ApplicationManagerInterface
{
    /**
     * @var \Application\Core\Model\Application[]
     */
    protected $enabled_apps;

    /**
     * @var \Application\Core\Model\Application[]
     */
    protected $installed_apps;

    /**
     * {@inheritdoc}
     */
    public function allEnabled($site)
    {
        $site_id = $this->getSiteId($site);

        if ($this->enabled_apps === null) {
            $options = [
                'select'     => '`from`.*',
                'joins'      => 'INNER JOIN ' . ApplicationSite::tableName() . ' AS appsite ON `from`.id = appsite.app_id',
                'conditions' => [
                    'appsite.site_id = :site',
                    'site' => $site_id,
                ],
                'order'      => '`from`.load_order',
            ];

            $this->enabled_apps[$site_id] = $this->getApps($options);
        }

        return $this->enabled_apps[$site_id];
    }

    /**
     * {@inheritdoc}
     */
    public function allInstalled()
    {
        if ($this->installed_apps === null) {
            $options = [
                'order' => 'load_order',
            ];

            $this->installed_apps = $this->getApps($options);
        }

        return $this->installed_apps;
    }

    /**
     * {@inheritdoc}
     */
    public function isInstalled($app_name)
    {
        return array_key_exists($app_name, $this->allInstalled());
    }

    /**
     * {@inheritdoc}
     */
    public function isEnabled($app_name, $site)
    {
        return array_key_exists($app_name, $this->allEnabled($site));
    }

    /**
     * {@inheritdoc}
     */
    public function createApplication($app_name)
    {
        if ($this->findByAppName($app_name) === null) {
            Application::create([
                'app_name'   => $app_name,
                'load_order' => Application::max('load_order') + 10,
            ]);
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteApplication($app_name)
    {
        $app = $this->findByAppName($app_name);

        if ($app && $app->total_installations === 0) {
            $app->delete();
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function createApplicationSite($app_name, $site)
    {
        if ($app = $this->findByAppName($app_name)) {
            ApplicationSite::create([
                'app_id'  => $app->id,
                'site_id' => $this->getSiteId($site),
            ]);
        } else {
            throw new ApplicationManagerNotFoundException();
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteApplicationSite($app_name, $site)
    {
        $options = [
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . Application::tableName() . ' AS a ON a.id = `from`.app_id',
            'conditions' => [
                'a.app_name = :name AND `from`.site_id = :site',
                'name' => $app_name,
                'site' => $this->getSiteId($site),
            ],
        ];

        if ($app_site = ApplicationSite::first($options)) {
            $app_site->delete();
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function findByAppName($app_name)
    {
        $apps = $this->allInstalled();

        if (array_key_exists($app_name, $apps)) {
            return $apps[$app_name];
        }

        return null;
    }

    /**
     * @return \Application\Core\Model\Application[]
     */
    protected function getApps(array $options = [])
    {
        $apps = [];
        /** @var \Application\Core\Model\Application $app */
        foreach (Application::all($options) as $app) {
            $apps[$app->getName()] = $app;
        }

        return $apps;
    }

    /**
     * Resetta la cache dei modelli
     */
    protected function clearCache()
    {
        $this->enabled_apps = null;
        $this->installed_apps = null;
    }

    /**
     * @param int|SiteInterface $site
     * @return int
     */
    private function getSiteId($site)
    {
        if ($site instanceof SiteInterface) {
            return $site->getId();
        } elseif (is_numeric($site)) {
            return (int) $site;
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'Expected an integer or an instance of SiteInterface in %s. Given "%s" instead.',
                    __METHOD__,
                    var_to_string($site)
                )
            );
        }
    }
}
