<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\AccountInterface;
use Application\Core\Model\Account;
use Application\Core\Model\AccountPortfolio;
use Application\Core\Model\Permit;
use Application\Core\Model\PermitRole;
use Application\Core\Model\Role;

class AccountManager implements AccountManagerInterface
{
    /** @var array */
    protected $identities = [];

    /**
     * {@inheritdoc}
     * @return Account
     */
    public function findById($id)
    {
        if (!isset($this->identities[$id])) {
            $this->identities[$id] = Account::find(intval($id));
        }

        return $this->identities[$id];
    }

    /**
     * {@inheritdoc}
     * @return Account
     */
    public function findByEmail($email)
    {
        if (!$email) {
            return null;
        }

        return Account::first([
            'conditions' => ['email = ?', $email],
        ]);
    }

    /**
     * {@inheritdoc}
     * @return Account
     */
    public function findByUsername($username)
    {
        if (!$username) {
            return null;
        }

        return Account::first([
            'conditions' => ['username = ?', $username],
        ]);
    }

    /**
     * {@inheritdoc}
     * @return Account
     */
    public function findByResetPasswordToken($token)
    {
        if (!$token) {
            return null;
        }

        return Account::first([
            'conditions' => ['reset_password_key = ?', $token],
        ]);
    }

    /**
     * @param string $url
     * @return Account
     */
    public function findByUrl($url)
    {
        return Account::find(['conditions' => ['url = :url OR old_url = :url', 'url' => $url]]);
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByIds(array $ids)
    {
        $checkedIds = [];
        foreach ($ids as $id) {
            $id = (int) $id;

            if (!$id) {
                continue;
            }

            $checkedIds[] = $id;
        }

        if ($checkedIds === []) {
            return [];
        }

        $queryIds = implode(', ', $checkedIds);

        $options = [
            'conditions' => [
                "id IN ({$queryIds})",
            ],
        ];

        return Account::all($options);
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByPermission($permissionKey, $orderBy = null, $limit = 50, $offset = 0)
    {
        $options = [
            'select'     => '`from`.*',
            'joins'      => implode(' ', [
                'INNER JOIN ' . PermitRole::tableName() . ' AS pr ON pr.role_id = `from`.role_id',
                'INNER JOIN ' . Permit::tableName() . ' AS p ON p.id = pr.permit_id',
            ]),
            'conditions' => [
                'p.key = :key AND p.is_enabled = :enabled',
                'key'     => $permissionKey,
                'enabled' => true,
            ],
            'limit'      => $limit,
            'offset'     => $offset,
        ];

        if ($orderBy) {
            $options['order'] = $orderBy;
        }

        return Account::all($options);
    }

    /**
     * {@inheritdoc}
     */
    public function search($keyword, array $options = [], $orderBy = null, $limit = 50, $offset = 0)
    {
        $options = array_merge([
            'searchableFields' => ['email', 'username', 'name', 'surname', 'company'],
            'active'           => true,
            'founder'          => false,
        ], $options);

        if (!is_array($options['searchableFields'])) {
            throw new \InvalidArgumentException('The option "searchableFields" must be an array.');
        }

        if (empty($options['searchableFields'])) {
            return [];
        }

        $searchConditions = [];
        foreach ($options['searchableFields'] as $field) {
            $searchConditions[] = "pongho_like({$field}, :term)";
        }

        $searchCondition = implode(' OR ', $searchConditions);

        $options = [
            'conditions' => [
                "is_active = :active AND is_founder = :founder AND role_id <> :anonymous AND ({$searchCondition})",
                'active'    => (bool) $options['active'],
                'founder'   => (bool) $options['founder'],
                'anonymous' => Role::USER_NOT_LOGGED,
                'term'      => "%{$keyword}%",
            ],
            'limit'      => $limit,
            'offset'     => $offset,
        ];

        if ($orderBy) {
            $options['order'] = $orderBy;
        }

        return Account::all($options);
    }

    /**
     * {@inheritdoc}
     * @return Account[]
     */
    public function findAncestorsByDescendant($descendant, $current_account = null, $can_view_all_users = true)
    {
        // Ricavo l'id dell'account per il quale estrarre i genitori
        if ($descendant instanceof AccountInterface) {
            $descendant_id = $descendant->getId();
        } else {
            $descendant_id = intval($descendant);
        }

        $ancestors_options = [];

        $joins = [];
        $conditions = [
            '`from`.id NOT IN(:anonymous, :self)',
            'anonymous' => Account::ANONYMOUS,
            'self'      => $descendant_id,
        ];

        if ($current_account !== null) {
            // Ricavo l'id dell'utente
            if ($current_account instanceof AccountInterface) {
                $current_account_id = $current_account->getId();
            } else {
                $current_account_id = intval($current_account);
            }

            $joins[] = 'LEFT JOIN ' . AccountPortfolio::tableName() . ' AS descendant ON descendant.descendant_id = `from`.id';

            // Gli utenti presenti in questo recordset devono essere discendenti dell’utente corrente
            if (!$can_view_all_users) {
                $conditions[0] .= ' AND descendant.ancestor_id = :current_user';
                $conditions['current_user'] = $current_account_id;
            }

            // todo: verificare coerenza commento con condizione della query
            // Gli utenti presenti in questo recordset NON devono essere discendenti dell’utente che si sta modificando
            $conditions[0] .= ' AND descendant.ancestor_id <> :account';
            $conditions['account'] = $current_account_id;
        }

        // Gli utenti presenti in questo pannello devono avere il permesso 'is_ancestor'
        $joins[] = 'LEFT JOIN ' . PermitRole::tableName() . ' AS pr ON pr.role_id = `from`.role_id';
        $joins[] = 'LEFT JOIN ' . Permit::tableName() . ' AS p ON p.id = pr.permit_id';

        $conditions[0] .= ' AND p.`key` = :permit_key';
        $conditions['permit_key'] = 'users.is_ancestor';
        // Compongo la query
        $ancestors_options['select'] = '`from`.*';
        $ancestors_options['joins'] = implode(' ', $joins);

        $ancestors_options['conditions'] = $conditions;

        return Account::all($ancestors_options);
    }

    public function save(AccountInterface $account)
    {
        $account->save();
    }
}
