<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\LanguageModuleInterface;
use Application\Core\Model\Manager\LanguageSiteManagerInterface;
use Application\Core\Model\Manager\ModuleManagerInterface;
use Application\Core\Model\Manager\ModuleSiteManager;
use Pongho\Core\Kernel;

/**
 * Modello per le relazioni tra lingue e moduli.
 *
 * @property int    $language_site_id
 * @property int    $module_id
 * @property string $path
 * @property string $name
 * @property int    $position
 * @property bool   $is_enabled
 */
class LanguageModule extends Base implements LanguageModuleInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'languages_modules';

    /**
     * Chiave primaria.
     *
     * @var string|array
     */
    public static $primary_key = ['language_site_id', 'module_id'];

    /**
     * Callback `after_save`.
     *
     * @var array
     */
    public static $after_save = ['updateLinkTypes'];

    /**
     * Modulo associato.
     *
     * @var ModuleSite
     */
    private $module;

    /**
     * Lingua associata.
     *
     * @var LanguageSite
     */
    private $language;

    /**
     * Ricerca in base a `language_id`, `site_id` e `module_id`.
     *
     * @param int $language_id
     * @param int $site_id
     * @param int $module_id
     * @return LanguageModule
     *
     * @deprecated
     */
    public static function findByLanguageAndSiteAndModule($language_id, $site_id, $module_id)
    {
        return self::first([
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id',
            'conditions' => [
                'ls.language_id = :language AND ls.site_id = :site AND `from`.module_id = :module',
                'language' => $language_id,
                'site'     => $site_id,
                'module'   => $module_id,
            ],
        ]);
    }

    /**
     * Ricerca in base a `language_site_id` e `module_id`.
     *
     * @param int $language_site_id
     * @param int $module_id
     * @return LanguageModule
     *
     * @deprecated
     */
    public static function findByLanguageSiteAndModule($language_site_id, $module_id)
    {
        return self::first([
            'conditions' => [
                'language_site_id = :language_site AND module_id = :module',
                'language_site' => $language_site_id,
                'module'        => $module_id,
            ],
        ]);
    }

    /**
     * Ricerca in base a `language_id` e `site_module_id`.
     *
     * @param int $language_id
     * @param int $site_module_id
     * @return LanguageModule
     *
     * @deprecated
     */
    public static function findByLanguageAndSiteModule($language_id, $site_module_id)
    {
        return self::first([
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . ModuleSite::tableName() . ' AS ms ON ms.module_id = `from`.module_id'
                . ' INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id AND ls.site_id = ms.site_id',
            'conditions' => [
                'ms.id = :site_module AND ls.language_id = :language',
                'site_module' => $site_module_id,
                'language'    => $language_id,
            ],
        ]);
    }

    /**
     * Ricerca tutti in base a `module_id` e `site_id`.
     *
     * @param int $module_id
     * @param int $site_id
     * @return LanguageModule[]
     *
     * @deprecated
     */
    public static function allByModuleAndSite($module_id, $site_id)
    {
        return self::all([
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id',
            'conditions' => [
                '`from`.module_id = :module AND ls.site_id = :site',
                'module' => $module_id,
                'site'   => $site_id,
            ],
        ]);
    }

    /**
     * Restituisce il modello `ModuleSite` associato.
     *
     * @throws \Exception Quando l’attributo `site_id` non è presente nel modello corrente.
     * @return ModuleSite
     *
     * @deprecated
     */
    public function getSiteModule()
    {
        if ($this->module === null) {
            if ($this->attributePresent('site_id')) {
                $this->module = ModuleSite::first([
                    'select'     => 'm.*, lm.*, `from`.*',
                    'joins'      => 'INNER JOIN ' . Module::tableName() . ' AS m ON m.id = `from`.module_id'
                        . ' INNER JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.module_id = m.id',
                    'conditions' => [
                        '`from`.module_id = :module AND `from`.site_id = :site AND lm.language_site_id = :language_site',
                        'module'        => $this->module_id,
                        'site'          => $this->site_id,
                        'language_site' => $this->language_site_id,
                    ],
                ]);
            } else {
                throw new \Exception('Not yet implemented!');
            }
        }

        return $this->module;
    }

    /**
     * Restituisce il modello `LanguageSite` associato.
     *
     * @return LanguageSite
     *
     * @deprecated
     */
    public function getSiteLanguage()
    {
        if ($this->language === null) {
            $this->language = LanguageSite::first([
                'select'     => 'l.*, `from`.*',
                'joins'      => 'INNER JOIN ' . Language::tableName() . ' AS l ON l.id = `from`.language_id',
                'conditions' => ['`from`.id = :languagesite', 'languagesite' => $this->language_site_id],
            ]);
        }

        return $this->language;
    }

    /**
     * {@inheritdoc}
     */
    public function getModuleId()
    {
        return $this->module_id;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguageSiteId()
    {
        return $this->language_site_id;
    }

    /**
     * {@inheritdoc}
     */
    public function getPath()
    {
        return $this->path;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * {@inheritdoc}
     */
    public function getPosition()
    {
        return $this->position;
    }

    /**
     * {@inheritdoc}
     */
    public function isEnabled()
    {
        return $this->is_enabled;
    }

    /**
     * {@inheritdoc}
     */
    public function enable()
    {
        $this->is_enabled = true;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function disable()
    {
        $this->is_enabled = false;

        return $this;
    }

    /**
     * Aggiorna il menu con i nuovi permalink.
     */
    public function updateLinkTypes()
    {
        $language_id = $this->getLanguage()->getId();

        // todo: manager
        foreach (LinkType::findByLanguageAndModelIdAndLinkType($language_id, $this->module_id, 'module') as $item) {
            $item->model_path = $this->path;
            $item->save();
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getModule()
    {
        /** @var ModuleManagerInterface $manager */
        $manager = Kernel::instance()->getContainer()->get('module_manager');

        return $manager->findById($this->module_id);
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguageSite()
    {
        /** @var LanguageSiteManagerInterface $manager */
        $manager = Kernel::instance()->getContainer()->get('language_site_manager');

        return $manager->findById($this->language_site_id);
    }

    /**
     * {@inheritdoc}
     */
    public function getModuleSite()
    {
        /** @var ModuleSiteManager $manager */
        $manager = Kernel::instance()->getContainer()->get('module_site_manager');

        $site_id = $this->getLanguageSite()->getSiteId();
        $module_site = $manager->findBySiteAndModule($site_id, $this->module_id);

        if (!$module_site) {
            throw new \InvalidArgumentException(
                sprintf(
                    'The module site with site "%s" and module "%s" is undefined',
                    $site_id,
                    $this->module_id
                )
            );
        }

        return $module_site;
    }

    /**
     * {@inheritdoc}
     */
    public function getSite()
    {
        return $this->getLanguageSite()->getSite();
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguage()
    {
        return $this->getLanguageSite()->getLanguage();
    }
}
