<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\SiteInterface;
use Locale;
use Pongho\Core\Kernel;

/**
 * Modello per le lingue.
 *
 * @property int    $id
 * @property string $iso
 * @property string $culture
 * @property string $name
 * @property string $locale
 * @property string $timezone
 */
class Language extends Base implements LanguageInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'languages';

    /**
     * Restituisce tutte le lingue abilitate del sito indicato.
     *
     * @param int $site_id
     * @return Language[]
     *
     * @deprecated
     */
    public static function findAllBySite($site_id)
    {
        return static::all([
            'select'     => 'ls.*, ls.id AS language_site_id, `from`.*',
            'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.language_id = `from`.id',
            'conditions' => [
                'ls.site_id = :site AND ls.is_enabled = :enabled',
                'site'    => $site_id,
                'enabled' => true,
            ],
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getIso()
    {
        return $this->iso;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * {@inheritdoc}
     */
    public function getCulture()
    {
        return $this->culture;
    }

    /**
     * {@inheritdoc}
     */
    public function getLocale()
    {
        return $this->locale;
    }

    /**
     * {@inheritdoc}
     */
    public function getTimezone()
    {
        return $this->timezone;
    }

    /**
     * {@inheritdoc}
     */
    public function startLanguage()
    {
        setlocale(LC_TIME, $this->locale);
        date_default_timezone_set($this->timezone);

        if (class_exists(Locale::class)) {
            Locale::setDefault('it_IT');
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguageSite($site)
    {
        if (is_numeric($site)) {
            $site_id = intval($site);
        } elseif ($site instanceof SiteInterface) {
            $site_id = $site->getId();
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'Site must be an integer or an instance of SiteInterface. "%s" given instead.',
                    var_to_string($site)
                )
            );
        }

        /** @var Manager\LanguageSiteManagerInterface $languageSiteManager */
        $languageSiteManager = Kernel::instance()->getContainer()->get('language_site_manager');

        return $languageSiteManager->findByLanguageAndSite($this->id, $site_id);
    }
}
